<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class OPENAPPGW_CustomProduct {
    private $name;
    private $shippingClass;
    private $weight;
    private $needsShipping;

    public function __construct($productArray) {
        $this->name = isset($productArray['name']) ? $productArray['name'] : '';
        $this->shippingClass = isset($productArray['shippingClass']) ? $productArray['shippingClass'] : '';
        $this->weight = isset($productArray['weight']) ? $productArray['weight'] : 0;
        $this->needsShipping = isset($productArray['needsShipping']) ? $productArray['needsShipping'] : false;
    }

    public function get_name() {
        return $this->name;
    }
    public function get_weight() {
        return $this->weight;
    }
    public function needs_shipping(){
        return $this->needsShipping;
    }
    public function get_shipping_class(){
        return $this->shippingClass;
    }
}

class OPENAPPGW_OpenApp_Gateway extends WC_Payment_Gateway {
    protected $api_key;
    protected $secret;
    protected $merchant_id;
    protected $profile_id;
    protected $open_app_url;

    protected $payment_method_title;

    private $basket_change_triggered = false;

    private $cart_id_to_process = null;

    private $supported_country = 'PL';

    private $shipping_methods;

    public function __construct() {
        $this->id = 'openapp';
        $this->icon = '';
        $this->has_fields = false;
        $this->method_title = __('OpenApp Gateway', 'openapp-gateway-for-woocommerce');
        $this->method_description = __('Convenient online payments through a mobile app', 'openapp-gateway-for-woocommerce');
        $this->payment_method_title = 'OpenApp';
        // other fields
        $this->api_key = $this->get_option('api_key');
        $this->secret = $this->get_option('secret');
        $this->merchant_id = $this->get_option('merchant_id');
        $this->profile_id = $this->get_option('profile_id');
        $this->open_app_url = $this->get_option('open_app_url');

        // Gateways can support subscriptions, refunds, saved payment methods.
        $this->supports = array(
            'products'
        );

        $this->shipping_methods = array(
            ''                      => __('Disabled','openapp-gateway-for-woocommerce'),
            'INPOST_APM'            => __('InPost Paczkomat','openapp-gateway-for-woocommerce'),
            'ORLEN_APM'             => __('Delivery with Orlen Paczka','openapp-gateway-for-woocommerce'),
            'POCZTA_POLSKA_APM'     => __('Delivery with Poczta Polska pickup (machines, pickup points)','openapp-gateway-for-woocommerce'),
            'DHL_PICKUP'            => __('Pickup from DHL pickup point','openapp-gateway-for-woocommerce'),
            'DPD_PICKUP'            => __('Pickup from DPD pickup point','openapp-gateway-for-woocommerce'),
            'INSTORE_PICKUP'        => __('Self-pickup by customer in your store','openapp-gateway-for-woocommerce'),
            'INPOST_COURIER'        => __('Delivery with InPost courier','openapp-gateway-for-woocommerce'),
            'DHL_COURIER'           => __('Delivery using DHL courier','openapp-gateway-for-woocommerce'),
            'DPD_COURIER'           => __('Delivery with DPD courier','openapp-gateway-for-woocommerce'),
            'UPS_COURIER'           => __('Delivery with UPS courier','openapp-gateway-for-woocommerce'),
            'FEDEX_COURIER'         => __('Delivery with FEDEX courier','openapp-gateway-for-woocommerce'),
            'GLS_COURIER'           => __('Delivery with GLS courier','openapp-gateway-for-woocommerce'),
            'POCZTEX_COURIER'       => __('Delivery with POCZTEX courier','openapp-gateway-for-woocommerce'),
            'GEIS_COURIER'          => __('Delivery with GEIS courier','openapp-gateway-for-woocommerce'),
            'ELECTRONIC'            => __('Electronic delivery, eg. license, cinema tickets','openapp-gateway-for-woocommerce')
        );



        $this->init_form_fields();

        // Load the settings.
        $this->init_settings();
        $this->title = $this->get_option( 'title' );
        $this->description = $this->get_option( 'description' );

        add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );

        add_filter('woocommerce_shipping_instance_form_fields_flat_rate', array($this,'add_special_key_field_to_methods'));
        add_filter('woocommerce_shipping_instance_form_fields_free_shipping', array($this,'add_special_key_field_to_methods'));
        add_filter('woocommerce_shipping_instance_form_fields_local_pickup', array($this,'add_special_key_field_to_methods'));

        // integration for: https://pl.wordpress.org/plugins/inpost-paczkomaty/
        add_filter( 'woocommerce_shipping_instance_form_fields_inpost_paczkomaty',array($this,'add_special_key_field_to_methods'));

        // integration for: https://pl.wordpress.org/plugins/flexible-shipping/
        add_filter( 'woocommerce_shipping_instance_form_fields_flexible_shipping_single',array($this,'add_special_key_field_to_methods'));


        // @TODO - maybe move to other place (?)
        add_action('oa_update_cart_in_db', array($this, 'store_cart_in_db'), 10, 1);

        add_action('shutdown', array($this, 'on_shutdown'), 21);
    }


    public function init() {

        // Check if the payment gateway is enabled
        if ( ! $this->is_available() ) {
            return;
        }

        // If the 'q' parameter is set (calling static file), return immediately
        $callingStaticFile = isset($_GET['q']) ? sanitize_text_field($_GET['q']) : null;

        if (!empty($callingStaticFile)) {
            return;
        }

        // If the request is a WP cron job, return immediately
        if (defined('DOING_CRON') && DOING_CRON) {
            return;
        }

        // actions for wp-admin
        $this->initAdminActions();

        if ( is_admin() ) {
            return;
        }

        // @TODO - move to other place (?)
        add_action('wp_enqueue_scripts', array($this,'oa_plugin_enqueue_scripts'));

        $this->registerAPIRoutes();
        $this->registerCartStorage();
        $this->registerOaOrder();

        // Check if the OALogin is enabled
        if ('yes' === $this->get_option('oalogin_enabled', 'no')) {
            $this->registerOaLogin();
        }

    }

    public function initAdminActions(){
        add_action('woocommerce_order_status_changed', array($this, 'oa_status_changed'), 10, 4);
        add_action('current_screen', array($this,'sse_support_script') );
        add_action('wp_ajax_test_sse_support', array($this,'test_sse_support_callback'));
        add_action('wp_ajax_sse_save_test_result', array($this,'handle_sse_save_test_result'));

        // order-returns
        add_action('add_meta_boxes', array($this, 'add_order_returns_meta_box'));
        add_action('wp_ajax_openapp_handle_return_action', array($this, 'handle_return_action_ajax'));

        add_filter('rest_pre_serve_request', array($this, 'prevent_caching'), 10, 4);

    }

    /**
     * ===========================================================================
     * 1. WooCommerce Payment gateway
     * ===========================================================================
     */
    public function init_form_fields(){
        $this->form_fields = array(
            'enabled' => array(
                'title'       => __('Enable/Disable','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable OpenApp Gateway','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => '',
                'default'     => 'no'
            ),
            'title' => array(
                'title'       => __('Title','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('This controls the title which the user sees during checkout.','openapp-gateway-for-woocommerce'),
                'default'     => __('OpenApp Payment','openapp-gateway-for-woocommerce'),
                'desc_tip'    => true,
            ),
            'description' => array(
                'title'       => __('Description','openapp-gateway-for-woocommerce'),
                'type'        => 'textarea',
                'description' => __('This controls the description which the user sees during checkout.','openapp-gateway-for-woocommerce'),
                'default'     => __('Pay with OpenApp payment gateway.','openapp-gateway-for-woocommerce'),
            ),
            //...other settings
            'api_key' => array(
                'title'       => __('API Key','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('Enter your OpenApp API key here.','openapp-gateway-for-woocommerce'),
                'default'     => '',
            ),
            'secret' => array(
                'title'       => __('Secret','openapp-gateway-for-woocommerce'),
                'type'        => 'password',
                'description' => __('Enter your OpenApp secret here.','openapp-gateway-for-woocommerce'),
                'default'     => '',
            ),
            'merchant_id' => array(
                'title'       => __('Merchant ID','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('Enter your OpenApp merchant ID here.','openapp-gateway-for-woocommerce'),
                'default'     => '',
            ),
            'profile_id' => array(
                'title'       => __('Profile ID','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('Enter your OpenApp profile ID here.','openapp-gateway-for-woocommerce'),
                'default'     => '',
            ),
            'open_app_url' => array(
                'title'       => __('API Base Url','openapp-gateway-for-woocommerce'),
                'type'        => 'text',
                'description' => __('Enter OpenApp API base url here.','openapp-gateway-for-woocommerce'),
                'default'     => 'https://api.uat.open-pay.com/merchant',
            ),
            'interval_time' => array(
                'title'       => __('Interval Time','openapp-gateway-for-woocommerce'),
                'type'        => 'select',
                'description' => __('Select the interval time for order redirection checking (AJAX polling). A smaller interval provides a faster redirection experience but may increase server load.','openapp-gateway-for-woocommerce'),
                'default'     => '8500',
                'options'     => array(
                    '2000'  => __('2 seconds','openapp-gateway-for-woocommerce'),
                    '5000'  => __('5 seconds','openapp-gateway-for-woocommerce'),
                    '8500'  => __('8.5 seconds','openapp-gateway-for-woocommerce')
                ),
            ),
            'sse_enabled' => array(
                'title'       => __('Server-Sent Events (SSE)','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable SSE for Order Redirection','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => sprintf(
                    __('When enabled, the plugin utilizes Server-Sent Events (SSE) to manage order check and thank you page redirection. This method is more efficient than traditional AJAX polling. Before enabling SSE, you can check your server\'s compatibility by clicking on the link: %s', 'openapp-gateway-for-woocommerce'),
                    '<a href="#" id="test-sse-button">' . __('Test SSE Support', 'openapp-gateway-for-woocommerce') . '</a><span id="ct-sse-test-result" style="margin-left:5px;"></span>'
                ),
                'default'     => 'no',
            ),
            'order_status' => array(
                'title'       => __('Order Status','openapp-gateway-for-woocommerce'),
                'type'        => 'select',
                'description' => __('Select the default order status for new orders, example: Processing','openapp-gateway-for-woocommerce'),
                'default'     => 'wc-processing',
                'options'     => wc_get_order_statuses()
            ),
            'basket_sync' => array(
                'title'       => __('Cart synchronization','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable cart synchronization','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => __('Activating this option will synchronize every change made to the WooCommerce shopping cart with the OpenApp mobile app in real-time.','openapp-gateway-for-woocommerce'),
                'default'     => 'no',
            ),
            'oalogin_enabled' => array(
                'title'       => __('Enable/Disable OALogin','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable OALogin','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => __('If checked, the OALogin functionality will be enabled. Uncheck this option to disable OALogin.','openapp-gateway-for-woocommerce'),
                'default'     => 'yes',
            ),
            'validation_enabled' => array(
                'title'       => __('Enable/Disable Validation','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable request validation','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => __('If checked, the plugin will validate OpenApp incoming requests. Disable this option only for testing purposes.','openapp-gateway-for-woocommerce'),
                'default'     => 'yes',
            ),
            'debug' => array(
                'title'       => __('Debug','openapp-gateway-for-woocommerce'),
                'label'       => __('Enable logging','openapp-gateway-for-woocommerce'),
                'type'        => 'checkbox',
                'description' => __('If checked, the plugin will log debug information to the PHP error log.','openapp-gateway-for-woocommerce'),
                'default'     => 'no',
            ),
        );
    }

    public function payment_fields() {}

    public function process_payment( $order_id ) {
        global $woocommerce;

        $order = wc_get_order( $order_id );
        $woocommerce->cart->empty_cart();
        return array(
            'result'   => 'success',
            'redirect' => $this->get_return_url( $order )
        );
    }
    public function is_available() {
        $is_available = ('yes' === $this->get_option('enabled', 'no'));
        return $is_available;
    }

    public static function is_sse_enabled() {
        // Retrieve the SSE support status from the transient
        $sseSupportStatus = get_transient('openappgw_sse_supported');

        // Check if the server supports SSE
        $serverSupportsSSE = ($sseSupportStatus !== false && $sseSupportStatus === "1");

        // Retrieve the WooCommerce option and check if SSE is enabled
        $options = get_option('woocommerce_openapp_settings');
        $sseEnabled = isset($options['sse_enabled']) && $options['sse_enabled'] === 'yes';

        // Return true only if both server supports SSE and the SSE option is enabled
        return $serverSupportsSSE && $sseEnabled;
    }

    public function sse_support_script() {
        $page = isset($_GET['page']) ? sanitize_text_field($_GET['page']) : '';
        $tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        $section = isset($_GET['section']) ? sanitize_text_field($_GET['section']) : '';

        if (!empty($page) && !empty($tab) && !empty($section) &&
            $page === 'wc-settings' &&
            $tab === 'checkout' &&
            $section === 'openapp') {
            add_action( 'admin_enqueue_scripts', array($this,'enqueue_sse_test_script') );
        }
    }

    public function enqueue_sse_test_script() {
        wp_enqueue_script( 'openappgw-admin-sse-test', OPENAPPGW_PLUGIN_DIR_URL . 'assets/js/sse-test.js', array(), OPENAPPGW_WOOCOMMERCE_GATEWAY, true );

        wp_localize_script('openappgw-admin-sse-test', 'sseTestParams', array(
            'ajaxUrlTest' => admin_url('admin-ajax.php?action=test_sse_support&security=' . wp_create_nonce('openappgw_sse_support_nonce')),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('openappgw_sse_support_nonce')
        ));

    }

    public function test_sse_support_callback() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this feature.'));
        }

        check_ajax_referer('openappgw_sse_support_nonce', 'security');

        // Set the headers for SSE
        header('Content-Type: text/event-stream');
        header('Cache-Control: no-cache');
        header('Connection: keep-alive');
        header('X-Accel-Buffering: no');

        // Run a loop to send messages periodically
        for ($i = 0; $i < 3; $i++) {
            echo "event: test\n";
            echo "data: " . wp_json_encode(array('message' => 'SSE test message')) . "\n\n";
            // Flush the output buffer to the client
            if (ob_get_level() > 0) ob_flush();
            flush();
            // Sleep for a second to simulate a delay between messages
            sleep(1);
        }

        // Close the connection after the test
        echo "event: close\n";
        echo "data: " . wp_json_encode(array('message' => 'Test complete')) . "\n\n";
        if (ob_get_level() > 0) ob_flush();
        flush();

        // Terminate the script without displaying the shutdown HTML
        exit(0);
    }

    public function handle_sse_save_test_result() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this feature.'));
        }

        check_ajax_referer('openappgw_sse_support_nonce', 'security');

        // Directly compare with the string 'true' or 'false'
        // Compare with the string 'true'
        $sseSupported = (isset($_POST['sseSupported']) && $_POST['sseSupported'] === 'true');
        // Cast the boolean to an integer (true to 1, false to 0)
        $sseSupportedInt = (int) $sseSupported;

        set_transient('openappgw_sse_supported', $sseSupportedInt, 0);

        wp_send_json_success(__('SSE support status updated.','openapp-gateway-for-woocommerce'));
    }

    public function prevent_caching($served, $result, $request, $server) {
        $route = $request->get_route();

        if (strpos($route, '/openapp/') !== false) {
            header("Cache-Control: no-cache, no-store, must-revalidate");
            header("Pragma: no-cache");
            header("Expires: 0");
        }

        return $served;
    }

    /**
     * Shipping methods
     */

    private function convertToCents($value) {
        // Handle null/empty values
        if ($value === null || $value === '') {
            return 0;
        }

        // Replace comma with dot for decimal separator (only comma/dot for cents)
        $normalized = str_replace(',', '.', (string) $value);

        // Convert to float, multiply by 100, and round with explicit mode
        return (int) round((float) $normalized * 100, 0, PHP_ROUND_HALF_UP);
    }

    private function rebuild_cart_contents_with_data($cart_data) {
        // Extract simplified cart contents and cart contents data from $cart_data
        $simplified_cart_contents = $cart_data['cart_contents'];
        $cart_contents_data = $cart_data['cart_contents_data'];

        $rebuilt_cart_contents = [];

        foreach ($simplified_cart_contents as $cart_item_key => $simplified_cart_item) {
            // If there is corresponding 'data' for this cart item, add it back
            if (isset($cart_contents_data[$cart_item_key])) {
                $customProduct = new OPENAPPGW_CustomProduct($cart_contents_data[$cart_item_key]);
                $simplified_cart_item['data'] = $customProduct;
            }

            // Add the rebuilt cart item to the array
            $rebuilt_cart_contents[$cart_item_key] = $simplified_cart_item;
        }

        return $rebuilt_cart_contents;
    }

    public function calculate_shipping_cost_for_cart($cart_contents_record, $method, $basket_value) {

        try {
            $woocommerce = WC();

            if ( ! isset($woocommerce->session)) {
                return null;
            }

            if ( null === $woocommerce->cart ) {
                $woocommerce->cart = new WC_Cart();
            }

            $woocommerce->cart->empty_cart();

            $cart_data = $this->rebuild_cart_contents_with_data($cart_contents_record);

            foreach ($cart_data as $item) {
                $_product = $this->get_product_object($item);
                if(is_null($_product)){
                    continue;
                }

                $product_id = $_product->get_id();
                $quantity = $item['quantity'];
                $woocommerce->cart->add_to_cart($product_id, $quantity);
            }

            $package = array(
                'destination' => array(
                    'country' => $this->supported_country,
                ),
                'contents' => $cart_data,
                'applied_coupon' => [],
                'contents_cost' => $basket_value
            );

            $method->calculate_shipping($package);

            $rates = $method->rates;
            $total_shipping_cost = 0;

            foreach ($rates as $rate_id => $rate) {
                $rate_cost = $rate->get_cost();
                $total_shipping_cost += $rate_cost;
            }

            return $this->convertToCents($total_shipping_cost);
        } catch (Exception $e) {
            // Handle exception
            $this->log_debug('Error occurred: ' . $e->getMessage());
            // You can return a specific value or even re-throw the exception
            return null; // Or any other appropriate response
        }
    }

    public function get_available_shipping_methods($country_code, $basket_value, $cart_contents_record = array()) {
        $cheapest_methods = array();

        // Check if WooCommerce is activated
        if (class_exists('WC_Shipping_Zones')) {
            // Get all the shipping zones
            $zones = WC_Shipping_Zones::get_zones();

            foreach ($zones as $zone) {

                $matching_locations = array_filter($zone['zone_locations'], function($location) use ($country_code) {
                    return $location->code == $country_code;
                });

                if (!empty($matching_locations)) {

                    $shipping_zone = new WC_Shipping_Zone($zone['id']);
                    $methods = $shipping_zone->get_shipping_methods(true);

                    foreach ($methods as $method) {
                        $oa_shipping_method_key = $method->get_option('oa_shipping_mapping');

                        if(!$oa_shipping_method_key){
                            continue;
                        }

                        $cost = $this->calculate_shipping_cost_for_cart($cart_contents_record, $method, $basket_value);


                        if(is_null($cost)){
                            continue;
                        }

                        $min_amount = $this->convertToCents($method->get_option('min_amount'));

                        if (!$min_amount || $basket_value >= $min_amount) {
                            if (!isset($cheapest_methods[$oa_shipping_method_key]) || $cost < $cheapest_methods[$oa_shipping_method_key]['cost']) {
                                $cheapest_methods[$oa_shipping_method_key] = array(
                                    'key' => $oa_shipping_method_key,
                                    'cost' => (int) $cost
                                );
                            }
                        }
                    }
                }
            }
        }

        $enabled_methods = array_values($cheapest_methods);
        return $enabled_methods;
    }

    public function add_special_key_field_to_methods($fields) {

        $fields['oa_shipping_mapping'] = array(
            'title'         => __('OpenApp mapping', 'openapp-gateway-for-woocommerce'),
            'type'          => 'select',
            'description'   => __('Select shipping method used in OpenApp mobile app', 'openapp-gateway-for-woocommerce'),
            'options'       => $this->shipping_methods,
            'default'       => '',
        );

        return $fields;
    }


    /**
     * ===========================================================================
     * 2. Initialize functionality
     * ===========================================================================
     */
    private function registerAPIRoutes(){
        add_filter( 'woocommerce_is_rest_api_request',  array($this,'exclude_specific_endpoint_from_rest') );

        add_action('rest_api_init', array($this, 'register_openapp_routes_external'));
        add_action('rest_api_init', array($this, 'register_openapp_routes_internal'));
    }
    private function registerCartStorage(){
        add_action('woocommerce_cart_loaded_from_session', array($this,'store_previous_cart_hash_in_session'), 99, 1);
        add_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));
    }

    private function registerOaOrder(){
        add_action('woocommerce_after_cart_table', array($this, 'openapp_qr_order_as_action'), 1);
        add_action('woocommerce_review_order_before_payment', array($this, 'openapp_qr_order_as_action'), 1);
        add_action('wp', array($this,'register_oa_order_shortcode'));
        add_action('woocommerce_thankyou_' . $this->id, array($this, 'oa_reset_order_key'), 2);
    }

    private function registerOaLogin(){
        add_action('wp_loaded', array($this, 'set_guest_session'));
        add_action('init', array($this,'oa_custom_login'));

        add_action('woocommerce_login_form_end', array($this,'openapp_qr_login_as_action'));
        add_action('wp', array($this,'register_oa_login_shortcode'));
    }

    /**
     * ===========================================================================
     * 3. registerAPIRoutes
     * ===========================================================================
     */

    public function exclude_specific_endpoint_from_rest( $is_rest_api_request ) {
        if ( empty( $_SERVER['REQUEST_URI'] ) ) {
            return $is_rest_api_request;
        }

        // Sanitize the REQUEST_URI
        $uri = sanitize_text_field( $_SERVER['REQUEST_URI'] );

        $uri_path = wp_parse_url($uri, PHP_URL_PATH);

        // Check if the path matches your specific endpoint
        if ( strpos($uri_path, '/openapp/v1/basket') !== false ) {
            return false;
        }

        return $is_rest_api_request;
    }

    public function register_openapp_routes_external() {

        register_rest_route(
            'openapp/v1',
            '/basket',
            array(
                'methods'  => 'POST',
                'callback' => array($this, 'create_or_update_basket'),
                'permission_callback' => '__return_true',
            )
        );


        register_rest_route(
            'openapp/v1',
            '/basket',
            array(
                'methods' => 'GET',
                'callback' => array($this,'retrieve_basket'),
                'permission_callback' => '__return_true',
                'args' => array(
                    'basketId' => array(
                        'validate_callback' => function($param, $request, $key) {
                            return !empty($param);
                        }
                    ),
                ),
            )
        );

        register_rest_route('openapp/v1', '/order', array(
            'methods' => 'POST',
            'callback' => array($this,'create_new_wc_order'),
            'permission_callback' => '__return_true',
        ));

        register_rest_route('openapp/v1', '/basket_recalculate', array(
            'methods' => 'POST',
            'callback' => array($this,'basket_recalculate'),
            'permission_callback' => '__return_true',
        ));

        register_rest_route( 'openapp/v1', '/identity', array(
            'methods' => 'POST',
            'callback' => array($this,'handle_identity'),
            'permission_callback' => '__return_true',
        ) );


        register_rest_route(
            'openapp/v1',
            '/products',
            array(
                'methods' => 'GET',
                'callback' => array($this,'retrieve_products'),
                'permission_callback' => '__return_true',
                'args' => array(
                    'full' => array(
                        'required' => false
                    )
                ),
            )
        );

        register_rest_route(
            'openapp/v1',
            '/order-return',
            array(
                'methods'  => 'POST',
                'callback' => array($this, 'handle_order_return'),
                'permission_callback' => '__return_true',
            )
        );
    }

    public function register_openapp_routes_internal(){
        register_rest_route('openapp/v1', '/qr_code', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_qr_code_data'),
            'args' => array(
                'cart_id' => array(
                    'required' => true
                )
            ),
            'permission_callback' => '__return_true',
        ));

        register_rest_route('openapp/v1', '/oa_redirection', array(
            'methods' => 'GET',
            'callback' => array($this, 'oa_check_order_redirection'),
            'args' => array(
                'cart_id' => array(
                    'required' => true
                )
            ),
            'permission_callback' => '__return_true',
        ));

        register_rest_route('openapp/v1', '/oa_login', array(
            'methods' => 'GET',
            'callback' => array($this, 'oa_check_login'),
            'args' => array(
                'cart_id' => array(
                    'required' => true
                )
            ),
            'permission_callback' => '__return_true',
        ));
    }

    /**
     * ===========================================================================
     * 4. registerCartStorage
     * ===========================================================================
     */

    private function generate_cart_hash(){
        $session_id = $this->get_session_id();
        $cart_contents = null;

        if ( function_exists('WC') && isset(WC()->cart) ) {
            $cart_contents = WC()->cart->get_cart_contents();
        }

        // Include the session_id when generating the hash
        $hash = md5(wp_json_encode($cart_contents) . $session_id);
        return $hash;
    }

    public function store_previous_cart_hash_in_session($cart){
        if ($this->is_request_empty()) {
            return;
        }

        if($this->is_heartbeat()){
            return;
        }

        $new_hash = $this->generate_cart_hash();
        $old_hash = WC()->session->get('previous_cart_hash');

        if ($old_hash !== $new_hash) {
            // Update session with new hash
            WC()->session->set('previous_cart_hash', $new_hash);
        }
    }


    private function is_request_empty() {
        return empty($_REQUEST);
    }

    private function generate_oa_basket_id($length = 10) {
        $bytes = random_bytes(ceil($length / 2));
        return substr(bin2hex($bytes), 0, $length);
    }

    private function get_unique_oa_basket_id(){
        global $wpdb;
        $max_attempts = 3;
        $attempts = 0;

        do {
            $cart_id = $this->generate_oa_basket_id();
            $existing_row = $wpdb->get_row($wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $cart_id));
            $attempts++;

            if ($attempts >= $max_attempts && $existing_row) {
                // Log an error if unable to generate a unique cart_id
                error_log('OA: Unable to generate unique cart_id after ' . $max_attempts . ' attempts.');
                return false;
            }
        } while ($existing_row);

        return $cart_id;
    }

    // asynchronous function (in theory: page can be refreshed faster then function completion)
    public function store_cart_in_db($force_rebuild = false)
    {
        if ($this->is_request_empty() && !$force_rebuild) {
            return;
        }

        $is_internal_api = isset($_SERVER['HTTP_X_WP_INTERNAL']) && sanitize_text_field($_SERVER['HTTP_X_WP_INTERNAL']) == 'true';

        if($is_internal_api){
            return;
        }

        $start_time = microtime(true);

        $new_hash = $this->generate_cart_hash();
        $old_hash = WC()->session->get('previous_cart_hash');

        if(!$force_rebuild && ($old_hash === $new_hash)){
           return;
        }



        global $wpdb;

        $session_id = $this->get_session_id();
        $hashed_session_id = hash('md5', $session_id);
        $applied_coupons = WC()->cart->get_applied_coupons();
        $coupon_data = array();

        foreach ($applied_coupons as $coupon_code) {
            $discount_amount = WC()->cart->get_coupon_discount_amount($coupon_code, WC()->cart->display_cart_ex_tax);
            $coupon_data[] = array(
                'code' => $coupon_code,
                'discount_amount' => $discount_amount,
            );
        }

        if(is_user_logged_in()) {
            $userId = strval(get_current_user_id());
        } else {
            $userId = md5(uniqid(microtime() . wp_rand(), true));
        }

        $my_cart = WC()->cart->get_cart();
        $cart_content_data = $this->get_cart_contents_data($my_cart);
        $cart_content = $this->get_simplified_cart_contents($my_cart);

        $cart_data = array(
            'cart_contents' => $cart_content,
            'coupon_data' => $coupon_data,
            'user_id' => $userId,
            'cart_contents_data' => $cart_content_data
        );

        $cart_data_serialized = maybe_serialize($cart_data);

        $existing_row = $wpdb->get_row($wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_session_id = %s", $hashed_session_id));

        if ($existing_row) {
            $wpdb->update(
                $wpdb->prefix . "oa_woocommerce_persistent_cart",
                array('cart_contents' => $cart_data_serialized, 'cart_expiry' => time() + 60 * 60 * 24 * 30),  // expires in 30 days
                array('cart_session_id' => $hashed_session_id),
                array('%s', '%s', '%d'),
                array('%s')
            );

            if (isset($existing_row->cart_id)) {
                $this->cart_id_to_process = $existing_row->cart_id;
                $this->basket_change_triggered = true;
            }
        } else {
            try {
                $cart_id = $this->get_unique_oa_basket_id();
                // set cart_id on new record creation
                $wpdb->insert(
                    $wpdb->prefix . "oa_woocommerce_persistent_cart",
                    array('cart_id' => $cart_id, 'cart_contents' => $cart_data_serialized, 'cart_session_id' => $hashed_session_id, 'cart_expiry' => time() + 60 * 60 * 24 * 30),  // expires in 30 days
                    array('%s', '%s', '%s', '%d')
                );
            } catch (Exception $e) {
                return;
            }
        }

        // End timing
        $end_time = microtime(true);
        $execution_time = round($end_time - $start_time, 4);
        $this->log_debug("store_cart_in_db: ".$execution_time);

    }

    public function on_shutdown() {

        if ('no' === $this->get_option('basket_sync', 'no')) {
            return;
        }

        if ($this->basket_change_triggered && $this->cart_id_to_process) {
            $uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : 'CLI or unknown';
            $method = isset($_SERVER['REQUEST_METHOD']) ? $_SERVER['REQUEST_METHOD'] : 'Unknown method';

            // Check for a POST request applying a coupon
            if ($method === 'POST' && strpos($uri, 'wc-ajax=apply_coupon') !== false) {
                return; // Exit early if applying a coupon
            }

            // Continue with POST requests

            // Check for specific GET requests
            if ($method === 'GET') {
                if (!(strpos($uri, 'removed_item') !== false || strpos($uri, 'undo_item') !== false)) {
                    return; // Exit early for GET requests that do NOT involve removing or undoing items
                }
            }

            remove_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));
            $this->oa_basket_changed($this->cart_id_to_process);
            add_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

            $this->basket_change_triggered = false;
            $this->cart_id_to_process = null;
        }
    }

    private function oa_basket_changed($basketId){
        if(!$basketId){
            return;
        }

        $endpoint = '/v1/basket/change';
        $method = 'POST';
        $url = $this->open_app_url . $endpoint;
        $context = 'openapp_basket_changed';

        $start_time = microtime(true);

        // Retrieve the basket data from your database
        $basket_data = $this->get_basket_data($basketId);

        if (empty($basket_data)) {
            return new WP_Error('basket_not_found', 'Basket not found', array('status' => 404));
        }

        $body = wp_json_encode($basket_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        $setupData = $this->setupRequest($endpoint, $method, $body, $context);
        $response = $this->executeRequest($url, $setupData, $context);

        $end_time = microtime(true);
        $execution_time = round($end_time - $start_time, 4);
        $this->log_debug("oa_basket_changed: ".$execution_time);

    }

    public function oa_status_changed($order_id, $from_status, $to_status, $order){
        $endpoint = '/v1/orders/multiFulfillment';
        $method = 'POST';
        $url = $this->open_app_url . $endpoint ;
        $context = 'openapp_basket_fulfillment';


        // Mapping WooCommerce statuses to the ones expected by OpenApp
        $status_mapping = array(
            'completed' => 'DELIVERED',
            'processing' => 'FULFILLED',
            'shipped' => 'SHIPPED',
            'ready-for-pickup' => 'READY_FOR_PICKUP',
            'cancelled' => 'CANCELLED_MERCHANT'
        );

        // Check if the changed status exists in our mapping
        if (!isset($status_mapping[$to_status])) {
            return; // Exit if we don't need to send data for this status
        }
        // Retrieve oaOrderId from the order's post meta
        $oaOrderId = get_post_meta($order_id, 'oaOrderId', true);

        if (!$oaOrderId) {
            return;
        }

        /**
         * get shipping method
         */
        $order = wc_get_order($order_id);
        if ( ! is_a( $order, 'WC_Order' ) ) {
            return;
        }

        $shipments = [];

        // Get the shipping methods from the order
        $shipping_methods = $order->get_shipping_methods();

        foreach ($shipping_methods as $shipping_method) {
            // Construct each shipment based on the shipping method
            $method_id = $shipping_method->get_method_id();
            $shipment = [
                'shipmentId' => $order_id .'_'. $method_id,
                'status' => $status_mapping[$to_status],
                'notes' => '',
                'operator' => $method_id,
            ];

            $shipments[] = $shipment;
        }

        // Prepare the order data for the API request
        $order_data = [
            'oaOrderId' => $oaOrderId,
            'shopOrderId' => (string)$order_id,
            'shipments' => $shipments
        ];

        $body = $this->encodeJsonResponse($order_data);
        $setupData = $this->setupRequest($endpoint, $method, $body, $context);
        $response = $this->executeRequest($url, $setupData, $context);
    }

    private function setupRequest($endpoint, $method, $body, $context) {
        $timestamp = (int) (microtime(true) * 1000);
        $nonce = $timestamp;
        $path = strtoupper($endpoint);

        $bodyHash = hash('sha256', $body, true);
        $bodyHashBase64 = $this->base64Encode($bodyHash);
        $hmacSignature = $this->generateHmacSignature($method, $path, $timestamp, $nonce, $bodyHashBase64, null, null, $context);
        $authHeader = "hmac v1$" . $this->api_key . "$" . $method . "$" . $path . "$" . $timestamp . "$" . $nonce;

        $this->openappgw_custom_log("Request Authorization Header: " . $authHeader, $context);
        $this->openappgw_custom_log("myBody: ". var_export($body, true), $context);

        return [
            'method' => $method,
            'headers' => array(
                'Content-Type' => 'application/json',
                'authorization' => sanitize_text_field($authHeader),
                'x-app-signature' => sanitize_text_field($hmacSignature)
            ),
            'body' => $body
        ];
    }

    private function executeRequest($url, $setupData, $context) {
        $method = $setupData['method'] ?? 'POST';

        $request_args = array(
            'headers' => $setupData['headers'],
            'body' => $setupData['body'],
            'timeout' => 10,
            'method' => $method,
            'blocking' => true
        );

        $this->openappgw_custom_log("=== EXECUTING REQUEST ===", $context);
        $this->openappgw_custom_log("URL: " . $url, $context);
        $this->openappgw_custom_log("Method: " . $method, $context);

        $response = wp_remote_request($url, $request_args);

        $responseBody = wp_remote_retrieve_body($response);
        $http_code = wp_remote_retrieve_response_code($response);

        // Logging response
        $this->openappgw_custom_log("RESPONSE:", $context);
        $this->openappgw_custom_log("HTTP Code: " . $http_code, $context);
        $this->openappgw_custom_log("Response Body: " . print_r($responseBody, true), $context);

        if (in_array($http_code, [200, 204])) {
            $this->openappgw_custom_log("Successful response received", $context);
        } else {
            $this->openappgw_custom_log("Error response received", $context);
        }

        return $response;
    }



    /**
     * ===========================================================================
     * 5. registerQRCodes
     * ===========================================================================
     */

    public function openapp_qr_order_as_action(){
        echo wp_kses_post($this->openapp_qr_order());
    }

    public function openapp_qr_login_as_action() {
        echo wp_kses_post($this->openapp_qr_login());
    }

    /**
     * ===========================================================================
     * 6. registerShortcodes
     * ===========================================================================
     */

    public function register_oa_order_shortcode() {
        $is_internal_api = isset($_SERVER['HTTP_X_WP_INTERNAL']) && sanitize_text_field($_SERVER['HTTP_X_WP_INTERNAL']) == 'true';

        if (is_admin() || (defined('DOING_CRON') && DOING_CRON) || wp_doing_ajax() || (defined('REST_REQUEST') && REST_REQUEST) || $is_internal_api) {
            return;
        }

        // enable only on page OR post
        if(is_page() || is_single() || is_archive()){
            $sanitized_uri = filter_var($_SERVER['REQUEST_URI'], FILTER_SANITIZE_URL);
            $this->log_debug('Shortcode triggered by ' . $sanitized_uri);

            add_shortcode('openapp_qr_order', array($this, 'openapp_qr_order'));
        }
    }

    public function register_oa_login_shortcode(){
        $is_internal_api = isset($_SERVER['HTTP_X_WP_INTERNAL']) && sanitize_text_field($_SERVER['HTTP_X_WP_INTERNAL']) == 'true';

        if (is_admin() || (defined('DOING_CRON') && DOING_CRON) || wp_doing_ajax() || (defined('REST_REQUEST') && REST_REQUEST) || $is_internal_api) {
            return;
        }

        // enable only on page OR post
        if(is_page() || is_single() || is_archive()){
            $sanitized_uri = filter_var($_SERVER['REQUEST_URI'], FILTER_SANITIZE_URL);
            $this->log_debug('Shortcode triggered by ' . $sanitized_uri);

            add_shortcode('openapp_qr_login', array($this, 'openapp_qr_login'));
        }
    }

    public function oa_plugin_enqueue_scripts() {
        // Register the script
        wp_register_script('openappgw-js', 'https://static.prd.open-pay.com/dist/barcode-library/openapp.min.1.0.0.js', array('jquery'), '1.0.0', false);
    }


    /**
     * ===========================================================================
     * 7. registerOaOrder
     * ===========================================================================
     */


    public function oa_check_order_redirection(WP_REST_Request $request)
    {

        $cart_id = sanitize_text_field($request->get_param('cart_id'));
        $useSSE = sanitize_text_field($request->get_param('use_sse'));

        $response = $this->createResponse(['redirect' => false]);

        $thank_you_url = $this->get_thank_you_url($cart_id);

        if ($thank_you_url) {
            $response = $this->createResponse(['redirect' => true, 'url' => $thank_you_url]);
        }

        if($useSSE){
            header('Content-Type: text/event-stream');
            header('Cache-Control: no-cache');
            header('Connection: keep-alive');
            header('X-Accel-Buffering: no');

            // set_time_limit(0);
            ignore_user_abort(true);

            $lastEventId = 0;

            while (true) {
                while (ob_get_level() > 0) {
                    ob_end_flush();
                }
                flush();

                $thank_you_url = $this->get_thank_you_url($cart_id);
                $lastEventId++;

                $this->openappgw_custom_log($lastEventId . ' ' . $thank_you_url, 'sse');

                if ($thank_you_url) {
                    echo "id: " . esc_html($lastEventId) . "\n";
                    echo "event: orderUpdate\n";
                    echo 'data: ' . wp_json_encode(['redirect' => true, 'url' => $thank_you_url]) . "\n\n";
                } else {
                    echo "id: " . esc_html($lastEventId) . "\n";
                    echo "event: orderUpdate\n";
                    echo 'data: ' . wp_json_encode(['redirect' => false]) . "\n\n";
                }

                flush();

                // Break the loop if a URL is found
                if ($thank_you_url) {
                    break;
                }

                sleep(3);
            }

        } else {
            return $response;
        }
    }

    public function get_thank_you_url($cart_id) {
        global $wpdb;

        $result = $wpdb->get_row(
            $wpdb->prepare("SELECT order_key FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $cart_id)
        );

        if(!empty($result) && $result->order_key !== NULL) {
            // order_key column contains order_number
            $order_id = $result->order_key;
            if ($order_id) {
                $order_key = null;
                $order = wc_get_order($order_id);
                if ($order && !is_wp_error($order)) {
                    $order_key = $order->get_order_key();
                }

                if(!is_null($order_key)){
                    $checkout_url = trailingslashit(wc_get_checkout_url()); // ensures that URL ends with a slash
                    $thank_you_url = add_query_arg(['key' => $order_key, 'order-received' => $order_id], $checkout_url . 'order-received/');

                    // Reset order key
                    // $this->oa_reset_order_key($order_id, false);

                    // Validate the URL
                    if (filter_var($thank_you_url, FILTER_VALIDATE_URL) !== false) {
                        // The URL is valid, proceed with the redirect
                        return $thank_you_url;
                    }
                }
            }
        }

        return false;
    }

    public function oa_reset_order_key($order_id, $assignOrder = true) {
        global $wpdb;

        $this->log_debug("reset order key");

        // Set the order_key back to NULL
        $wpdb->update(
            $wpdb->prefix . "oa_woocommerce_persistent_cart",
            ['order_key' => NULL],
            ['order_key' => $order_id]  // Where order_key matches
        );

        if($assignOrder){
            // assign guest order to customer
            $order = wc_get_order($order_id);
            $user_email = $order->get_billing_email();
            $user = get_user_by('email', $user_email);

            if (!$user) {
                $random_password = wp_generate_password(16, true);
                $user_id = $this->createWpUser($user_email, $random_password);

                if (is_wp_error($user_id)) {

                }
            } else {
                $user_id = $user->ID;
            }
            $order->set_customer_id($user_id);
            $order->save();


            // Trigger status update in API
            $current_status = $order->get_status();
            $this->oa_status_changed($order_id, $current_status, $current_status, $order);
        }

    }

    /**
     * ===========================================================================
     * 8. registerOaLogin
     * ===========================================================================
     */

    public function set_guest_session(){

        $oaSetGuestSession = filter_input(INPUT_GET, 'oa-set-guest-session', FILTER_SANITIZE_NUMBER_INT);

        if($oaSetGuestSession && $oaSetGuestSession === '1'){
            if (!is_user_logged_in()) {
                if ( isset( WC()->session ) ) {
                    if ( !WC()->session->has_session() ) {
                        $this->log_debug("set_session_for_guest");
                        WC()->session->set_customer_session_cookie(true);

                        // user session is active
                        // rebuild data
                        $this->store_cart_in_db(true);
                    }
                }
            }
            $redirect_url = wp_get_referer() ? wp_get_referer() : home_url();
            wp_safe_redirect($redirect_url.'?oa-initialized=1');
            exit;
        }
    }


    public function oa_check_login(WP_REST_Request $request)
    {
        $cart_id = sanitize_text_field($request->get_param('cart_id'));

        $this->log_debug("oa_check_login");

        if (!empty($cart_id)) {
            global $wpdb;
            $cart_data = $wpdb->get_row(
                $wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $cart_id)
            );


            if (!is_null($cart_data) && isset($cart_data->oa_auth_token) && !is_null($cart_data->oa_auth_token)) {
                $args = array(
                    'meta_key'     => 'oa_auth_token',
                    'meta_value'   => $cart_data->oa_auth_token,
                    'meta_compare' => '=',
                    'number'       => 1,
                );

                $users = get_users($args);

                if (!empty($users) && $user = $users[0]) {
                    $this->log_debug("oa_check_login: ". $user->user_email);

                    return $this->createResponse(['should_login' => true, 'redirect_url' => '/?oa-custom-login=1']);
                }
            }
        }

        return $this->createResponse(['should_login' => false]);
    }

    public function oa_custom_login(){
        $oaCustomLogin = filter_input(INPUT_GET, 'oa-custom-login', FILTER_SANITIZE_NUMBER_INT);

        if($oaCustomLogin && $oaCustomLogin === "1"){

            $this->log_debug("INIT oa-custom-login..");

            // Redirect to my account page
            $myaccount_page_id = get_option( 'woocommerce_myaccount_page_id' );
            $myaccount_url = get_permalink( $myaccount_page_id );

            if ( is_user_logged_in() ) {
                // do nothing (?)
            }

            $sessionId = $this->get_session_id();
            $sessionId = $this->get_cart_id_by_session($sessionId);

            // check if oauth_token exists
            // if yes - use associated email to login current user
            global $wpdb;

            $cart_data = $wpdb->get_row(
                $wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $sessionId)
            );


            if (is_null($cart_data)) {
                return new WP_REST_Response( "Session do not exists", 500 );
            }

            if(isset($cart_data->oa_auth_token) && !is_null($cart_data->oa_auth_token)){
                $args = array(
                    'meta_key'     => 'oa_auth_token',
                    'meta_value'   => $cart_data->oa_auth_token,
                    'meta_compare' => '=',
                    'number'       => 1,
                );

                $users = get_users($args);

                if (!empty($users)) {
                    $user = $users[0];

                    if ($user) {
                        wp_clear_auth_cookie();
                        wp_set_current_user($user->ID, $user->user_login);
                        wp_set_auth_cookie($user->ID);
                        do_action('wp_login', $user->user_login, $user);

                        $this->log_debug("oa-custom-login: ". $user->user_email);

                        // clear auth token
                        $wpdb->query(
                            $wpdb->prepare("UPDATE " . $wpdb->prefix . "oa_woocommerce_persistent_cart SET oa_auth_token = NULL WHERE cart_id = %s", $sessionId)
                        );


                        // Update last login time
                        update_user_meta( $user->ID, 'oa_last_login', time() );

                        // Redirect to account page if the login cookie is already set.
                        wp_safe_redirect($myaccount_url);
                        exit;
                    }
                }
            }

            wp_safe_redirect($myaccount_url);
            exit;
        }
    }


    /**
     * ===========================================================================
     * 9. HMAC Authentication
     * ===========================================================================
     */

    private function hmacSignatureIsValid($authorizationHeader, $serverSignature, $responseBodyHashBase64 = null){
        // Split the HMAC header into its parts
        list($version, $api_key, $method, $path, $timestamp, $nonce) = explode('$', $authorizationHeader);

        // Check if the timestamp is within the allowed 60 seconds window
        $currentTimestamp = round(microtime(true) * 1000); // Current time in milliseconds
        $diff = abs($currentTimestamp - $timestamp);

        if ($diff > 60000) {
            return false; // Reject if the timestamp difference is greater than 60 seconds
        }

        $hmacSignature = $this->generateHmacSignature($method, $path, $timestamp, $nonce, $responseBodyHashBase64, $api_key, $this->secret);

        if(hash_equals($serverSignature, $hmacSignature)){
            return true;
        } else {
            return false;
        }
    }

    protected function generateHmacSignature($method, $path, $timestamp, $nonce, $content = null, $api_key = null, $secret = null, $log_to_context = '') {
        if(!$api_key){
            $api_key = $this->api_key;
        }
        if(!$secret){
            $secret = $this->secret;
        }

        $stringToSign = "v1$"."$api_key$" . "$method$" . "$path$" . "$timestamp$" . "$nonce";

        if ($content) {
            $stringToSign .= "$".$content;
        }

        if($log_to_context){
            $this->openappgw_custom_log("STRING_TO_SIGN: " . $stringToSign, $log_to_context);
        }


        $hmacHash = hash_hmac('sha256', $stringToSign, $secret, true);
        $hmacSignature = $this->base64Encode($hmacHash);

        return $hmacSignature;
    }


    private function calculate_server_authorization($headers, $responseBody = null) {
        $authorization = isset($headers["authorization"]) ? $headers["authorization"][0] : null;

        if(!is_null($authorization)) {
            $components = explode('$', $authorization);

            if (count($components) < 6) {
                // The authorization string doesn't contain as many components as we're expecting.
                return null;
            }


            $timestamp = $components[4];
            $nonce = $components[5];

            $calculatedSignature = $this->generateHmacResponseSignature($timestamp, $nonce, $responseBody, $authorization);

            $context = 'openapp_hmac_response';
            $this->openappgw_custom_log("calculatedSignature: ". $calculatedSignature, $context);

            return "hmac v1". "$"."$timestamp"."$".$nonce."$"."$calculatedSignature";
        }

        return null;
    }

    protected function calculate_x_server_auth($headers, $responseBody = null) {
        $authorization = isset($headers["x-server-authorization"]) ? $headers["x-server-authorization"] : null;

        if(!is_null($authorization)) {
            $components = explode('$', $authorization);

            if (count($components) < 4) {
                // The authorization string doesn't contain as many components as we're expecting.
                return null;
            }

            $timestamp = $components[1];
            $nonce = $components[2];

            $calculatedSignature = $this->generateHmacResponseSignature($timestamp, $nonce, $responseBody, $authorization);

            $context = 'openapp_hmac_response';
            $this->openappgw_custom_log("calculatedSignature X-Server-Auth: ". $calculatedSignature, $context);

            $hmacXServerHeader = "hmac v1". "$"."$timestamp"."$".$nonce."$"."$calculatedSignature";

            return $hmacXServerHeader;
        }

        return null;
    }


    protected function generateHmacResponseSignature($timestamp,$nonce,$responseBody = null, $authorization = null, $printDebug = false) {

        $stringToSign =  "v1$"."$timestamp"."$"."$nonce";

        if (!is_null($responseBody)) {
            // FIX (encode body to convert backslash characters)
            $responseBody = wp_json_encode($responseBody);

            $responseBodyHash = hash('sha256', $responseBody, true);

            $responseBodyHashBase64 = $this->base64Encode($responseBodyHash);
            $stringToSign .= "$" . $responseBodyHashBase64;
        }


        $context = 'openapp_hmac_response';
        $this->openappgw_custom_log("GENERATE_HMAC_RESPONSE_SIGNATURE", $context);
        $this->openappgw_custom_log("Request Authorization Header: ". $authorization, $context);
        $this->openappgw_custom_log("Timestamp: ".$timestamp, $context);
        $this->openappgw_custom_log("Nonce: ".$nonce, $context);
        $this->openappgw_custom_log("responseBody: ". var_export($responseBody, true), $context);
        if(isset($responseBodyHash)){
            $this->openappgw_custom_log("responseHash: ".$responseBodyHash, $context);
        }
        if(isset($responseBodyHashBase64)){
            $this->openappgw_custom_log("responseBodyHashBase64: ".$responseBodyHashBase64, $context);
        }

        $this->openappgw_custom_log("stringToSign: ".$stringToSign, $context);
        $this->openappgw_custom_log("-----------------------", $context);

        $hmacHash = hash_hmac('sha256', $stringToSign, $this->secret, true);
        $hmacHashBase64 = $this->base64Encode($hmacHash);

        if($printDebug){
            // Output
            var_dump("GENERATING X-Server-Authorization Header");
            var_dump("Secret: ". $this->mask_secret($this->secret));
            var_dump("bodyDigest: " . $responseBodyHashBase64);
            var_dump("Nonce: " . $nonce);
            var_dump("Signature: " . $hmacHashBase64 );
            var_dump("Timestamp: " . $timestamp);
            var_dump("StringToSign: " . $stringToSign);
        }

        return $hmacHashBase64;
    }



    private function isRequestValid($headers, $body = null) {
        if (!isset($headers['authorization'][0]) || !isset($headers['x_app_signature'][0])) {
            return false;
        }

        // Check the value of the validation_enabled option
        if ('no' === $this->get_option('validation_enabled', 'yes')) {
            return true;
        }

        $validHmac = $this->hmacSignatureIsValid($headers['authorization'][0], $headers['x_app_signature'][0], $body);

        $context = 'openapp_hmac_response';
        $this->openappgw_custom_log("isRequestValid Auth: ". var_export($headers['authorization'][0], true), $context);
        $this->openappgw_custom_log("isRequestValid X-App-Signature: ". var_export($headers['x_app_signature'][0], true), $context);
        $this->openappgw_custom_log("isRequestValid Body: ". var_export($body, true), $context);
        $this->openappgw_custom_log("isRequestValid: ". var_export($validHmac, true), $context);
        $this->openappgw_custom_log("---------------------", $context);

        return $validHmac;
    }


    /**
     * ===========================================================================
     * 10. REST API ENDPOINTS
     * ===========================================================================
     */

    public function retrieve_products(WP_REST_Request $request) {
        try {
            $full_data = $request->get_param('full');
            $headers = $request->get_headers();

            // Pagination parameters - only use if explicitly provided
            $page = $request->get_param('page');
            $per_page = $request->get_param('per_page');
            $use_pagination = !empty($page) || !empty($per_page);

            if ($use_pagination) {
                $page = (int) $page ?: 1;
                $per_page = (int) $per_page ?: 50;
            }

            // Logger
            $context = 'openapp_retrieve_products';
            $this->openappgw_custom_log("Full data: " . print_r($full_data, true), $context);
            $this->openappgw_custom_log("Pagination: page={$page}, per_page={$per_page}, use_pagination=" . ($use_pagination ? 'yes' : 'no'), $context);
            $this->openappgw_custom_log("Headers: " . print_r($this->encodeJsonResponse($headers), true), $context);

            // Verify HMAC
            $validHmac = $this->isRequestValid($headers);
            if (!$validHmac) {
                return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
            }

            // Fetch products efficiently
            $args = array(
                'limit' => -1,
                'type' => ['variable', 'simple']
            );
            $products = wc_get_products($args);
        } catch (Exception $e) {
            $this->openappgw_custom_log("Error in retrieve_products: " . $e->getMessage(), 'openapp_retrieve_products');
            return new WP_Error('product_retrieval_error', 'Failed to retrieve products: ' . $e->getMessage(), array('status' => 500));
        }
        try {
            $response = array();
            $current_count = 0;
            $items_added = 0;
            $offset = $use_pagination ? ($page - 1) * $per_page : 0;

            // Get units once for performance
            $weight_unit = get_option('woocommerce_weight_unit');
            $dimension_unit = get_option('woocommerce_dimension_unit');

            foreach ($products as $product) {
                if ($use_pagination && $items_added >= $per_page) {
                    break;
                }

                try {
                    // Check if the product is a variable product
                    if ($product->is_type('variable')) {
                        // Get the variations
                        $variations = $product->get_children();

                        foreach ($variations as $variation_id) {
                            if ($use_pagination && $current_count < $offset) {
                                $current_count++;
                                continue;
                            }

                            if ($use_pagination && $items_added >= $per_page) {
                                break;
                            }

                            try {
                                $_variation = wc_get_product($variation_id);
                                if ($_variation) {
                                    $response[] = $this->get_product_response($_variation, $full_data === 'yes', $weight_unit, $dimension_unit);
                                    $current_count++;
                                    if ($use_pagination) $items_added++;
                                }
                            } catch (Exception $e) {
                                $this->openappgw_custom_log("Error processing variation {$variation_id}: " . $e->getMessage(), $context);
                                // Continue processing other variations
                            }
                        }
                    } else {
                        if ($use_pagination && $current_count < $offset) {
                            $current_count++;
                            continue;
                        }

                        // For simple products
                        $response[] = $this->get_product_response($product, $full_data === 'yes', $weight_unit, $dimension_unit);
                        $current_count++;
                        if ($use_pagination) $items_added++;
                    }
                } catch (Exception $e) {
                    $product_id = $product ? $product->get_id() : 'unknown';
                    $this->openappgw_custom_log("Error processing product {$product_id}: " . $e->getMessage(), $context);
                    // Continue processing other products
                }
            }
        } catch (Exception $e) {
            $this->openappgw_custom_log("Error in product loop: " . $e->getMessage(), $context);
            return new WP_Error('product_processing_error', 'Failed to process products: ' . $e->getMessage(), array('status' => 500));
        }

        // Return response with or without pagination metadata
        if ($use_pagination) {
            // Calculate accurate total items for pagination
            $total_items = 0;
            foreach ($products as $product) {
                if ($product->is_type('variable')) {
                    $total_items += count($product->get_children());
                } else {
                    $total_items++;
                }
            }

            $total_pages = ceil($total_items / $per_page);

            $paginated_response = array(
                'data' => $response,
                'pagination' => array(
                    'page' => $page,
                    'per_page' => $per_page,
                    'total_items' => $total_items,
                    'total_pages' => $total_pages,
                    'has_next_page' => $page < $total_pages,
                    'has_prev_page' => $page > 1
                )
            );

            // Create WP_REST_Response object
            $wpRestResponse = new WP_REST_Response($paginated_response, 200);

            // Calculate X-Server-Authorization and set the header if it exists
            $expectedXServerAuth = $this->calculate_server_authorization($headers, $paginated_response);
            if ($expectedXServerAuth !== null) {
                $this->openappgw_custom_log("X-Server-Authorization: " . $expectedXServerAuth, $context);
                $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
            }

            return $wpRestResponse;
        } else {
            // Return all items without pagination wrapper

            // Create WP_REST_Response object
            $wpRestResponse = new WP_REST_Response($response, 200);

            // Calculate X-Server-Authorization and set the header if it exists
            $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);
            if ($expectedXServerAuth !== null) {
                $this->openappgw_custom_log("X-Server-Authorization: " . $expectedXServerAuth, $context);
                $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
            }

            return $wpRestResponse;
        }
    }

    // Helper function to format product/variation data
    private function get_product_response($_product, $full, $weight_unit = '', $dimension_unit = '') {
        // Get product images
        $images = array();
        foreach ($_product->get_gallery_image_ids() as $image_id) {
            $images[] = wp_get_attachment_image_url($image_id, 'full');
        }

        // Get main product image
        $main_image = wp_get_attachment_image_url($_product->get_image_id(), 'full');
        if ($main_image) {
            array_unshift($images, $main_image); // Make sure main image is first
        }

        // Prepare stock data
        // important - to have param edit
        $manage_stock = $_product->get_manage_stock('edit');
        $stock_quantity = $manage_stock ? $_product->get_stock_quantity() : null;

        // Get categories and tags (for full data only)
        // Determine if the product is a variation and get its parent
        $parent_id = $_product->is_type('variation') ? $_product->get_parent_id() : $_product->get_id();
        $parent_product = wc_get_product($parent_id);

        $categories = array();
        $tags = array();

        if ($full) {
            // Fetch category names from parent product
            $category_ids = $parent_product->get_category_ids();
            foreach ($category_ids as $category_id) {
                $category = get_term($category_id, 'product_cat');
                if (!is_wp_error($category)) {
                    $categories[] = $category->name;
                }
            }

            // Fetch tag names from parent product
            $tag_ids = $parent_product->get_tag_ids();
            foreach ($tag_ids as $tag_id) {
                $tag = get_term($tag_id, 'product_tag');
                if (!is_wp_error($tag)) {
                    $tags[] = $tag->name;
                }
            }
        }

        // Get description and short_description, fallback to parent if variation is empty
        $description = $_product->get_description();
        if (empty($description) && $_product->is_type('variation')) {
            $description = $parent_product->get_description();
        }

        $short_description = $_product->get_short_description();
        if (empty($short_description) && $_product->is_type('variation')) {
            $short_description = $parent_product->get_short_description();
        }

        if ($full) {
            return array(
                'id' => $_product->get_id(),
                'sku' => $_product->get_sku(),
                'ean' => $_product->get_global_unique_id(),
                'name' => $_product->get_name(),
                'type' => $_product->get_type(),
                'description' => $description,
                'short_description' => $short_description,
                'price' => $this->convertToCents($_product->get_price()),
                'regular_price' => $this->convertToCents($_product->get_regular_price()),
                'sale_price' => $this->convertToCents($_product->get_sale_price()),
                'stock_status' => $_product->get_stock_status(),
                'stock_quantity' => $stock_quantity,
                'weight' => $_product->get_weight(),
                'weight_unit' => $weight_unit,
                'length' => $_product->get_length(),
                'width' => $_product->get_width(),
                'height' => $_product->get_height(),
                'dimension_unit' => $dimension_unit,
                'permalink' => $_product->get_permalink(),
                'images' => $images,
                'categories' => $categories,
                'tags' => $tags,
                'attributes' => $_product->is_type('variation') ? $_product->get_attributes() : null, // Variation attributes
            );
        } else {
            return array(
                'id' => $_product->get_id(),
                'sku' => $_product->get_sku(),
                'ean' => $_product->get_global_unique_id(),
                'name' => $_product->get_name(),
                'price' => $this->convertToCents($_product->get_price()),
                'regular_price' => $this->convertToCents($_product->get_regular_price()),
                'sale_price' => $this->convertToCents($_product->get_sale_price()),
                'stock_status' => $_product->get_stock_status(),
                'stock_quantity' => $stock_quantity
            );
        }
    }

    public function retrieve_basket( WP_REST_Request $request) {
        // Fetch the basketId from the request
        $basketId = $request->get_param('basketId');

        // Fetch the headers from the request
        $headers = $request->get_headers();

        // logger
        $context = 'openapp_retrieve_basket';
        $this->openappgw_custom_log("Basket ID: ".print_r($basketId, true), $context);
        $this->openappgw_custom_log("Headers: ".print_r($this->encodeJsonResponse($headers), true), $context); // Logs the headers

        // Verify HMAC
        $validHmac = $this->isRequestValid($headers);

        if(!$validHmac){
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        // Retrieve the basket data from your database
        $response = $this->get_basket_data($basketId);

        if (empty($response)) {
            return new WP_Error('basket_not_found', 'Basket not found', array('status' => 404));
        }


        // Create WP_REST_Response object
        $wpRestResponse = new WP_REST_Response($response, 200);

        $this->openappgw_custom_log("responseBody: ". var_export($response, true), $context);

        // Calculate X-Server-Authorization and set the header if it exists
        $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);

        if($expectedXServerAuth !== null) {
            $this->openappgw_custom_log("X-Server-Authorization: ".$expectedXServerAuth, $context);
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }

        $this->openappgw_custom_log("------------------------------------", $context);

        return $wpRestResponse;
    }

    public function create_or_update_basket(WP_REST_Request $request) {
        /**
         * Do not store cart in DB
         */
        remove_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

        $headers = $request->get_headers();

        // Validate HMAC
        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if (!$validHmac) {
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        $data = $request->get_json_params();

        // Validate input model
        if (empty($data['basket']) || empty($data['basket']['products'])) {
            return new WP_Error('invalid_data', 'basket and products are required', array('status' => 400));
        }

        $basket = $data['basket'];
        $products = $basket['products'];
        $loggedUser = $basket['loggedUser'] ?? null;

        $currency = get_woocommerce_currency();
        $cart_id = $basket['id'] ?? $this->get_unique_oa_basket_id();

        // --- Reuse your recalc/init logic ---
        $output = $this->initialize_output($cart_id, $currency, $loggedUser);

        $adjusted_products = $this->adjust_quantities($products);

        $cart_contents_record = $this->build_cart_contents_record([
            'products' => $adjusted_products,
            'price' => [ 'discounts' => [] ], // Adjust if you want to support input discounts
            'loggedUser' => $loggedUser
        ]);

        $output['products'] = $adjusted_products;
        $output['price']['basketValue'] = $cart_contents_record['basketValue'];
        $output['price']['discounts'] = $cart_contents_record['applied_discounts'];

        $output['deliveryOptions'] = $this->get_available_shipping_methods(
            $this->supported_country,
            $cart_contents_record['basketValue'],
            $cart_contents_record
        );

        $wpRestResponse = new WP_REST_Response($output, 200);

        $expectedXServerAuth = $this->calculate_server_authorization($headers, $output);
        if ($expectedXServerAuth !== null) {
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }

        // Re-add the action after calculations are done
        // not sure, if it is needed..
        add_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

        return $wpRestResponse;
    }




    public function basket_recalculate(WP_REST_Request $request) {
        /**
         * Do not store cart in DB
         */
        remove_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

        $data = $request->get_json_params();
        $headers = $request->get_headers();

        $context = 'openapp_basket_recalculate';
        $this->openappgw_custom_log(print_r($data, true), $context);
        $this->openappgw_custom_log(print_r($this->get_values_and_types($data), true), $context);
        $this->openappgw_custom_log("Headers: ".print_r($this->encodeJsonResponse($headers), true), $context);

        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if(!$validHmac){
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        $currency = get_woocommerce_currency();
        $output = $this->initialize_output($data['id'], $currency, $data['loggedUser'] ?? null);

        // Adjust quantities based on stock, and calculate total value and products
        $adjusted_products = $this->adjust_quantities($data['products']);

        // Build cart_contents_record after adjusting quantities
        $cart_contents_record = $this->build_cart_contents_record([
            'products' => $adjusted_products,
            'price' => $data['price'],
            'loggedUser' => $data['loggedUser']
        ]);


        $output['products'] = $adjusted_products;
        $output['price']['basketValue'] =  $cart_contents_record['basketValue'];
        $output['price']['discounts'] =  $cart_contents_record['applied_discounts'];


        $output['deliveryOptions'] = $this->get_available_shipping_methods(
            $this->supported_country,
            $cart_contents_record['basketValue'],
            $cart_contents_record
        );


        $this->openappgw_custom_log("RESPONSE:", $context);
        $this->openappgw_custom_log(print_r($output, true), $context);
        $this->openappgw_custom_log("------------------------------------", $context);

        $wpRestResponse = new WP_REST_Response($output, 200);

        $expectedXServerAuth = $this->calculate_server_authorization($headers, $output);
        if ($expectedXServerAuth !== null) {
            $this->openappgw_custom_log("X-Server-Authorization: ".$expectedXServerAuth, $context);
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }


        // Re-add the action after calculations are done
        // not sure, if it is needed..
        add_action('woocommerce_cart_updated', array($this, 'store_cart_in_db'));

        return $wpRestResponse;
    }


    private function build_cart_contents_record($data)
    {
        // Create a temporary WC_Cart instance
        $fake_cart = new WC_Cart();

        // Add products to the fake cart
        foreach ($data['products'] as $product_data) {
            $product_id = $product_data['id'];
            $quantity = $product_data['quantity'];
            $variation_id = $product_data['variation_id'] ?? 0;
            $variation = $product_data['variation'] ?? [];

            // Add product to the fake cart
            $fake_cart->add_to_cart($product_id, $quantity, $variation_id, $variation);
        }

        $discounts = new WC_Discounts($fake_cart);

        // Apply each discount code from the data
        foreach ($data['price']['discounts'] as $discount) {
            $coupon_code = $discount['code'];
            $coupon = new WC_Coupon($coupon_code);

            // Use WC_Discounts->is_coupon_valid to check if the coupon is valid
            if ($discounts->is_coupon_valid($coupon)) {
                $fake_cart->apply_coupon($coupon_code);
            }
        }

        // Calculate totals to ensure discounts are applied
        $fake_cart->calculate_totals();

        // Retrieve cart_contents from the fake cart
        $cart_contents = $fake_cart->get_cart();

        // Prepare cart_contents_data with additional product info
        $cart_contents_data = [];
        foreach ($cart_contents as $key => $item) {
            $wc_product = wc_get_product($item['product_id']);
            if ($wc_product) {
                $cart_contents_data[$key] = [
                    'name' => $wc_product->get_name(),
                    'shippingClass' => $wc_product->get_shipping_class(),
                    'weight' => $wc_product->get_weight(),
                    'needsShipping' => $wc_product->needs_shipping()
                ];
            }
        }

        $applied_discounts = [];
        foreach ($fake_cart->get_applied_coupons() as $coupon_code) {
            $discount_amount = $fake_cart->get_coupon_discount_amount($coupon_code);

            $applied_discounts[] = [
                'code' => $coupon_code,
                'value' => $this->convertToCents($discount_amount)
            ];
        }

        $basketValue = $this->calculate_total_price($cart_contents);

        return [
            'cart_contents' => $cart_contents,
            'coupon_data' => $data['price']['discounts'],
            'user_id' => $data['loggedUser'] ?? null,
            'cart_contents_data' => $cart_contents_data,
            'basketValue' => $basketValue,
            'applied_discounts' => $applied_discounts
        ];
    }


    private function adjust_quantities($products) {
        $adjusted_products = [];

        foreach ($products as $product_data) {
            $wc_product = wc_get_product($product_data['id']);
            if (!$wc_product) continue;

            $quantity = $product_data['quantity'];
            $error = $this->check_stock_availability($wc_product, $quantity);

            // $quantity is already adjusted within check_stock_availability if needed
            $adjusted_products[] = $this->create_product_output_array($wc_product, $quantity, $error);
        }

        return $adjusted_products;
    }

    private function initialize_output($id, $currency, $loggedUser = null) {

        $cart_id = $id ?? $this->get_unique_oa_basket_id();

        $output = [
            'id' => $cart_id,
            'expiresAt' => gmdate('Y-m-d\TH:i:s\Z', strtotime('+1 day')),
            'price' => [
                'currency' => $currency,
                'discounts' => [],
                'basketValue' => 0
            ],
            'deliveryOptions' => [],
            'products' => []
        ];

        if ($loggedUser) {
            $output['loggedUser'] = $loggedUser;
        }

        return $output;
    }

    private function check_stock_availability($wc_product, &$quantity) {
        $error = null;

        // Check if the product is in stock
        if (!$wc_product->is_in_stock()) {
            $error = "OUT_OF_STOCK";
            $quantity = 0;
        } elseif ($wc_product->managing_stock()) {
            // If managing stock, check the stock quantity
            $stock_quantity = $wc_product->get_stock_quantity();
            if ($quantity > $stock_quantity) {
                $quantity = $stock_quantity;
                $error = "QUANTITY_TOO_BIG";
            }
        }

        return $error;
    }



    /**
    * Place order
     */
    public function create_new_wc_order(WP_REST_Request $request) {
        // Parse and validate request body
        $data = $request->get_json_params();

        // Fetch the headers from the request
        $headers = $request->get_headers();

        // logger
        $context = 'openapp_place_order';
        $this->openappgw_custom_log(print_r($data, true), $context);
        $this->openappgw_custom_log(print_r($this->get_values_and_types($data), true), $context);
        $this->openappgw_custom_log("Headers: ".print_r($this->encodeJsonResponse($headers), true), $context); // Logs the headers

        if (empty($data['oaOrderId'])) {
            return new WP_Error('missing_oaOrderId', 'OA Order ID is required', array('status' => 400));
        }

        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if(!$validHmac){
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        $cart_id = $data['basket']['id'];

        // Validate all products exist before creating the order
        $invalid_products = array();
        foreach ($data['basket']['products'] as $product) {
            $product_to_add = wc_get_product($product['id']);
            if (!$product_to_add) {
                $invalid_products[] = $product['id'];
            }
        }

        if (!empty($invalid_products)) {
            return new WP_REST_Response( array(
                'message' => 'Error: Product(s) with id(s) ' . implode(', ', $invalid_products) . ' do not exist',
            ), 400 );
        }

        // All products valid, now create order
        $order = wc_create_order();

        foreach ($data['basket']['products'] as $product) {
            $product_to_add = wc_get_product($product['id']);
            $order->add_product($product_to_add, $product['quantity']);
        }


        // Here, we apply the discounts
        if (!empty($data['basket']['price']['discounts'])) {
            foreach ($data['basket']['price']['discounts'] as $discount) {
                // You could implement a function that validates the discount before applying it
                if ($this->validate_discount($discount)) {
                    $coupon = new WC_Coupon($discount['code']);

                    // Create an instance of the WC_Discounts class.
                    $discounts = new WC_Discounts( $order );

                    // Check if the coupon is valid for this order.
                    $valid = $discounts->is_coupon_valid( $coupon );

                    if ( is_wp_error( $valid ) ) {
                        // The coupon is not valid. You can log the error message if you wish
                        // error_log( $valid->get_error_message() );
                    } else {
                        $order->apply_coupon($coupon->get_code());
                    }
                }
            }
        }

        /**
         * Save data start
         */
        // Update order details
        if (!empty($data['billingDetails'])) {
            $order->set_billing_company($data['billingDetails']['companyName']);
            $order->set_billing_first_name($data['billingDetails']['firstName']);
            $order->set_billing_last_name($data['billingDetails']['lastName']);
            $order->set_billing_address_1($data['billingDetails']['street']);

            $billing_address_1 = $data['billingDetails']['street'] . ' ' .$data['billingDetails']['streetNo'];
            if(!empty($data['billingDetails']['apartmentNo'])){
                $billing_address_1 .= ' / ' . $data['billingDetails']['apartmentNo'];
            }
            $order->set_billing_address_1($billing_address_1);
            $order->set_billing_city($data['billingDetails']['city']);
            $order->set_billing_postcode($data['billingDetails']['postalCode']);
            $order->set_billing_country($data['billingDetails']['country']);
            $order->set_billing_phone($data['billingDetails']['phoneNumber']);

            if(!empty($data['billingDetails']['notes'])){
                $order->set_customer_note($data['billingDetails']['notes']);
            }
        }


        // Save email
        $order->set_billing_email($data['deliveryDetails']['email']);

        // Set payment method
        $order->set_payment_method($this->id);
        $order->set_payment_method_title($this->payment_method_title);

        // Update shipping details
        // Concatenate street number and apartment number
        $address_line_1 = $data['deliveryDetails']['street'] . ' ' .$data['deliveryDetails']['streetNo'];
        if(!empty($data['deliveryDetails']['apartmentNo'])){
            $address_line_1 .= ' / ' . $data['deliveryDetails']['apartmentNo'];
        }
        $order->set_shipping_address_1($address_line_1);

        // $order->set_shipping_address_2('');
        $order->set_shipping_company($data['deliveryDetails']['companyName']);
        $order->set_shipping_first_name($data['deliveryDetails']['firstName']);
        $order->set_shipping_last_name($data['deliveryDetails']['lastName']);
        $order->set_shipping_phone($data['deliveryDetails']['phoneNumber']);
        $order->set_shipping_city($data['deliveryDetails']['city']);
        $order->set_shipping_postcode($data['deliveryDetails']['postalCode']);
        $order->set_shipping_country($data['deliveryDetails']['country']);

        // Paczkomaty
        if (isset($data['deliveryDetails']['method']) && $data['deliveryDetails']['method'] === 'INPOST_APM') {
            if(isset($data['deliveryDetails']['id'])) {
                $parcelLockerId = $data['deliveryDetails']['id'];
                $order->set_shipping_company($parcelLockerId);

                // Save the Parcel Locker ID
                update_post_meta($order->get_id(), 'paczkomat_key', $parcelLockerId);

                // Construct and save the full name and address of the parcel locker
                $parcelLockerFull = $data['deliveryDetails']['id'] . ', ' .
                    $data['deliveryDetails']['street'] . ' ' .
                    $data['deliveryDetails']['streetNo'] . ', ' .
                    $data['deliveryDetails']['postalCode'] . ' ' .
                    $data['deliveryDetails']['city'];
                update_post_meta($order->get_id(), 'Wybrany paczkomat', $parcelLockerFull);
            }
        }


        // save Notes + TaxID
        $taxIdNote = "";
        if(!empty($data['billingDetails']['taxId'])){
            $taxId = $data['billingDetails']['taxId'];
            $taxIdNote = "TaxId: ".$taxId . "\n";
            // Save NIP for Baselinker
            update_post_meta($order->get_id(), '_billing_nip', $taxId);
        }

        $deliveryNotes = isset($data['deliveryDetails']['notes']) ? $data['deliveryDetails']['notes'] : '';
        $billingNotes = isset($data['billingDetails']['notes']) ? $data['billingDetails']['notes'] : '';

        $order->add_order_note($taxIdNote . $deliveryNotes . "\n" . $billingNotes);

        // save note2
        $note2 = "OA Order ID: " . $data['oaOrderId'] . "\n";
        $note2 .= "Payment Currency: " . $data['paymentDetails']['currency'] . "\n";
        $note2 .= "Payment Amount: " . $data['paymentDetails']['amount'] . "\n";
        $note2 .= "Delivery Type: " . $data['deliveryDetails']['type'];
        $order->add_order_note($note2);

        // Update orderOAID
        update_post_meta($order->get_id(), 'oaOrderId', $data['oaOrderId']);


        /**
         * Save data end
         */

        // Delivery method
        $deliveryMethodKey = isset($data['deliveryDetails']['method']) ? $data['deliveryDetails']['method'] : null;
        $deliveryMethodName = isset($this->shipping_methods[$deliveryMethodKey]) ? $this->shipping_methods[$deliveryMethodKey] : $deliveryMethodKey;

        // Add the shipping method to the WooCommerce order
        $deliveryCost = isset($data['basket']['price']['deliveryCost']) ? $data['basket']['price']['deliveryCost'] : 0;
        $deliveryCost = $deliveryCost / 100; // converts groszy to zloty (adjust as necessary)

        // Baselinker - shipping method Paczkomaty
        $shippingItem = new WC_Order_Item_Shipping();
        $shippingItem->set_method_title($deliveryMethodName);
        $shippingItem->set_method_id($deliveryMethodKey);
        $shippingItem->set_total( $deliveryCost);

        $calculate_tax_for = array(
            'country' => $this->supported_country,
        );

        $taxesAmount = 0;

        if (wc_tax_enabled()) {
            $tax_rates = WC_Tax::find_shipping_rates($calculate_tax_for);
            $taxes = WC_Tax::calc_tax($deliveryCost, $tax_rates, true);
            $taxesAmount = array_sum($taxes); // Calculating sums of all taxes
        }

        $netDeliveryCost = $deliveryCost - $taxesAmount;
        $shippingItem->set_total($netDeliveryCost);
        $order->add_item($shippingItem);

        $order->calculate_totals();

        $order_id = $order->save();

        if($order_id){
            $max_return_days = 30;

            $response = array(
                "shopOrderId" => (string)$order_id,
                "returnPolicy" => array(
                    "maxReturnDays" => $max_return_days,
                )
            );

            // Increment order_count for the basket in your custom table
            global $wpdb;

            $wpdb->query(
                $wpdb->prepare(
                    "UPDATE " . $wpdb->prefix . "oa_woocommerce_persistent_cart SET order_count = order_count + 1, order_key = %s, oaOrderId = %s WHERE cart_id = %s",
                    $order_id,
                    $data['oaOrderId'],
                    $cart_id
                )
            );

            // Trigger payment complete
            $txn_id = $data['oaOrderId'] . '_' . date('ymd_His');
            $order->payment_complete($txn_id);

            /**
             * Once all processing is complete, update the order status
             */
            // Fetch the selected order status from the plugin settings
            $selected_status = $this->get_option('order_status', 'wc-processing');
            $selected_status_val = str_replace('wc-', '', $selected_status);

            // Update order status
            $order->update_status($selected_status_val);

            // Create WP_REST_Response object
            $wpRestResponse = new WP_REST_Response( $response, 200 );


            // Calculate X-Server-Authorization and set the header if it exists
            $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);
            // Set X-Server-Authorization header if it exists
            if($expectedXServerAuth !== null) {
                $this->openappgw_custom_log("X-Server-Authorization: ".$expectedXServerAuth, $context);
                $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
            }

            $this->openappgw_custom_log("------------------------------------", $context);

            return $wpRestResponse;

        }else{
            return new WP_REST_Response( 'Error in Order Creation', 500 );
        }
    }

    public function handle_identity( WP_REST_Request $request ) {
        // Parse and validate request body
        $data = $request->get_json_params();

        // Fetch the headers from the request
        $headers = $request->get_headers();

        // logger
        $context = 'openapp_identity';
        $this->openappgw_custom_log(print_r($data, true), $context);
        $this->openappgw_custom_log(print_r($this->get_values_and_types($data), true), $context);
        $this->openappgw_custom_log("Headers: ".print_r($this->encodeJsonResponse($headers), true), $context); // Logs the headers

        $this->openappgw_custom_log("------------------------------------", $context);

        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if(!$validHmac){
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        // Sanitize and get the parameters
        $email = sanitize_email($data[ 'email' ]);
        $token = sanitize_text_field($data[ 'token' ]);


        if(empty($email) || empty($token)){
            return new WP_Error('missing_email_token', 'Incorrect parameters', array('status' => 400));
        }
        // check if token (cart_id) exists
        global $wpdb;

        $cart_data = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $token)
        );

        if (is_null($cart_data)) {
            return new WP_REST_Response( "Session do not exists", 500 );
        }

        // check if token exists
        $auth_token = wp_generate_password(32, true, false);  // generates a 64-character random string

        // Get user by email
        $user = get_user_by( 'email', $email );

        if ( $user ) {
            $this->destroy_user_sessions( $user->ID );
            // If the user already exists, just update the token
            update_user_meta( $user->ID, 'oa_auth_token', $auth_token );

            // Update oa_auth_token
            $wpdb->query(
                $wpdb->prepare("UPDATE " . $wpdb->prefix . "oa_woocommerce_persistent_cart SET oa_auth_token = %s WHERE cart_id = %s", $auth_token, $token)
            );


            // update_user_meta( $user->ID, 'oa_last_login', time() );

        } else {
            // If the user does not exist, create a new user
            $random_password = wp_generate_password( 16, true );
            $user_id = $this->createWpUser($email, $random_password);

            // Check if the user was created successfully
            if ( ! is_wp_error( $user_id ) ) {
                $this->destroy_user_sessions( $user_id );
                // Update the token
                update_user_meta( $user_id, 'oa_auth_token', $auth_token );
                // Update oa_auth_token
                $wpdb->query(
                    $wpdb->prepare("UPDATE " . $wpdb->prefix . "oa_woocommerce_persistent_cart SET oa_auth_token = %s WHERE cart_id = %s", $auth_token, $token)
                );

            } else {
                // Handle error
                // For example, return a REST response with an error message
                return new WP_REST_Response( "Unable to create user.", 500 );
            }
        }

        $wpRestResponse = new WP_REST_Response( null, 200 );

        // Calculate X-Server-Authorization and set the header if it exists
        $expectedXServerAuth = $this->calculate_server_authorization($headers, null);

        if($expectedXServerAuth !== null) {
            $this->openappgw_custom_log("X-Server-Authorization: ".$expectedXServerAuth, $context);
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }

        return $wpRestResponse;
    }


    /**
     * ===========================================================================
     * Return handler
     * ===========================================================================
     */
    public function handle_order_return(WP_REST_Request $request) {
        $context = 'openapp_order_return';

        // Get request data
        $data = $request->get_json_params();
        $headers = $request->get_headers();

        // Log the incoming request
        $this->openappgw_custom_log('Return request received: ' . print_r($data, true), $context);
        $this->openappgw_custom_log('Headers: ' . print_r($this->encodeJsonResponse($headers), true), $context);

        // Validate HMAC
        $body = $request->get_body();
        $bodyHash = hash('sha256', $body, true);
        $responseBodyHashBase64 = $this->base64Encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);

        if (!$validHmac) {
            return new WP_Error('invalid_auth', 'Unauthorized request', array('status' => 403));
        }

        // Validate required fields
        if (empty($data['oaOrderId']) || empty($data['oaOrderReturnId']) || empty($data['shopOrderId'])) {
            return new WP_Error('missing_required_fields', 'Required fields missing', array('status' => 400));
        }

        // Find the WooCommerce order
        $shop_order_id = $data['shopOrderId'];
        $order = wc_get_order($shop_order_id);

        if (!$order) {
            return new WP_Error('order_not_found', 'Order not found', array('status' => 404));
        }

        // Verify the order belongs to this OA order
        $stored_oa_order_id = get_post_meta($shop_order_id, 'oaOrderId', true);
        if ($stored_oa_order_id !== $data['oaOrderId']) {
            return new WP_Error('order_mismatch', 'Order ID mismatch', array('status' => 400));
        }

        // Generate a unique case ID for this return
        $case_id = $this->generate_return_case_id($data);

        // Process the return
        $this->process_order_return($order, $data, $case_id);

        // Prepare response
        $response = array(
            'caseId' => $case_id
        );

        $this->openappgw_custom_log('Return processed successfully. Case ID: ' . $case_id, $context);

        // Create response with proper headers
        $wpRestResponse = new WP_REST_Response($response, 200);

        // Calculate and set X-Server-Authorization header
        $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);
        if ($expectedXServerAuth !== null) {
            $this->openappgw_custom_log("X-Server-Authorization: " . $expectedXServerAuth, $context);
            $wpRestResponse->set_headers(['X-Server-Authorization' => $expectedXServerAuth]);
        }

        return $wpRestResponse;
    }
    private function generate_return_case_id($return_data) {
        $order_id = $return_data['shopOrderId'];
        $date = date('ymd_His');
        return "OA-RET-{$order_id}-{$date}";
    }
    private function process_order_return($order, $return_data, $case_id) {
        $context = 'openapp_process_return';

        // Store return information as order meta
        $return_meta = array(
            'oa_return_id' => $return_data['oaOrderReturnId'],
            'case_id' => $case_id,
            'return_type' => $this->determine_return_type($return_data['returnedProducts']),
            'status' => 'RECEIVED',
            'return_data' => $return_data,
            'created_at' => current_time('mysql')
        );

        // HPOS-compatible way to store meta
        $order->update_meta_data('_openapp_return_' . $return_data['oaOrderReturnId'], $return_meta);
        $order->save();

        // Create order note with return details
        $note = $this->build_return_note($return_data, $case_id);
        $order->add_order_note($note);

        // Set order status based on return type and settings
        $this->update_order_status_for_return($order, $return_meta['return_type']);

        // Log the return processing
        $this->openappgw_custom_log('Return processed for order ' . $order->get_id(), $context);
    }
    private function determine_return_type($returned_products) {
        $types = array();

        foreach ($returned_products as $product) {
            if (isset($product['returnType'])) {
                $types[] = $product['returnType'];
            }
        }

        // Get unique types and return as comma-separated string
        $unique_types = array_unique($types);
        return implode(', ', $unique_types);
    }

    private function build_return_note($return_data, $case_id) {
        $note = "=== OPENAPP RETURN REQUEST ===\n";
        $note .= "Case ID: " . $case_id . "\n\n";

        $note .= $this->format_data_recursively($return_data);

        return $note;
    }

    private function format_data_recursively($data, $indent = 0) {
        $output = '';
        $spacing = str_repeat('  ', $indent);

        foreach ($data as $key => $value) {
            $label = $this->format_label($key);

            if (is_array($value)) {
                // Handle arrays of objects (like returnedProducts)
                if (isset($value[0]) && is_array($value[0])) {
                    $output .= $spacing . $label . ":\n";
                    foreach ($value as $index => $item) {
                        $output .= $spacing . "  " . ($index + 1) . ".\n";
                        $output .= $this->format_data_recursively($item, $indent + 2);
                        $output .= "\n";
                    }
                } else {
                    // Handle nested objects
                    $output .= $spacing . $label . ":\n";
                    $output .= $this->format_data_recursively($value, $indent + 1);
                }
            } else {
                // Handle single values
                $formatted_value = $this->format_value($key, $value);
                $output .= $spacing . $label . ": " . $formatted_value . "\n";
            }
        }

        return $output;
    }

    private function format_label($key) {
        // Convert camelCase/snake_case to readable labels
        $label = preg_replace('/([a-z])([A-Z])/', '$1 $2', $key);
        $label = str_replace('_', ' ', $label);
        return ucwords($label);
    }

    private function format_value($key, $value) {
        // Handle special formatting for known fields
        if (in_array($key, ['unitPrice', 'linePrice', 'totalPrice'])) {
            return wc_price($value / 100);
        }

        if ($key === 'id' && is_numeric($value)) {
            // Try to get product name for product IDs
            $product = wc_get_product($value);
            if ($product) {
                return $product->get_name() . " (ID: {$value})";
            }
        }

        if (is_bool($value)) {
            return $value ? 'Yes' : 'No';
        }

        if (is_null($value)) {
            return 'N/A';
        }

        return $value;
    }

    private function update_order_status_for_return($order, $return_type) {
        // Get setting for return handling
        $return_status_setting = $this->get_option('return_order_status', 'on-hold');

        // Handle multiple return types
        $types = array_map('trim', explode(',', $return_type));

        if (count($types) === 1) {
            $single_type = $types[0];
            switch ($single_type) {
                case 'CLAIM':
                    $order->update_status($return_status_setting, 'Return claim received - requires investigation');
                    break;
                case 'WITHDRAWAL':
                    $order->update_status($return_status_setting, 'Customer withdrawal request received');
                    break;
                default:
                    $order->update_status($return_status_setting, 'Return request received');
                    break;
            }
        } else {
            // Multiple types
            $message = 'Mixed return request received (' . $return_type . ')';
            $order->update_status($return_status_setting, $message);
        }
    }


    public function add_order_returns_meta_box() {
        $screen = get_current_screen();

        // Add meta box for different possible screen IDs
        $screens = array('shop_order', 'woocommerce_page_wc-orders', 'wc_orders');

        foreach ($screens as $screen_id) {
            add_meta_box(
                'openapp_returns_' . $screen_id,
                'OpenApp Returns',
                array($this, 'display_order_returns_meta_box'),
                $screen_id,
                'normal',
                'high'
            );
        }
    }

    public function display_order_returns_meta_box($post) {
        $order_id = $post->ID;
        $returns = $this->get_order_returns($order_id);

        if (empty($returns)) {
            echo '<p>No returns found for this order.</p>';
            return;
        }

        echo '<div class="openapp-returns-container">';
        echo '<style>
.return-status { padding: 4px 8px; border-radius: 3px; font-weight: bold; }
.status-received { background: #ffc107; color: #000; }
.status-accepted { background: #28a745; color: #fff; }
.status-rejected { background: #dc3545; color: #fff; }
.return-actions { margin-top: 15px; }
.return-details { margin: 15px 0; padding: 15px; background: #f9f9f9; border-left: 4px solid #0073aa; }
.return-data-section { margin-top: 15px; padding: 10px; background: #fff; border: 1px solid #ddd; border-radius: 4px; }
.return-data-section h5 { margin-top: 0; color: #333; }
.return-data-content { font-family: monospace; white-space: pre-line; font-size: 12px; line-height: 1.4; }
.openapp-button { margin-right: 10px !important; color: #fff !important; text-decoration: none !important; }
.openapp-button.accept { background: #28a745 !important; border-color: #28a745 !important; }
.openapp-button.reject { background: #dc3545 !important; border-color: #dc3545 !important; }
.openapp-button:hover.accept { background: #218838 !important; border-color: #1e7e34 !important; }
.openapp-button:hover.reject { background: #c82333 !important; border-color: #bd2130 !important; }
</style>';

        foreach ($returns as $return_meta_key => $return) {
            $this->display_single_return($return, $return_meta_key, $order_id);
        }

        echo '</div>';

        // Add JavaScript for AJAX handling
        $this->add_returns_admin_script();
    }

    private function display_single_return($return, $meta_key, $order_id) {
        $return_data = $return['return_data'];

        echo '<div class="return-details">';
        echo '<h4>Return Request: ' . esc_html($return['case_id']) . '</h4>';

        // Basic return info
        echo '<div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">';
        echo '<div>';
        echo '<p><strong>Return Type:</strong> ' . esc_html($return['return_type']) . '</p>';
        echo '<p><strong>Status:</strong> <span class="return-status status-' . strtolower($return['status']) . '">' . esc_html($return['status']) . '</span></p>';
        echo '<p><strong>OA Return ID:</strong> ' . esc_html($return['oa_return_id']) . '</p>';
        echo '</div>';
        echo '<div>';
        echo '<p><strong>Created:</strong> ' . esc_html($return['created_at']) . '</p>';
        if (isset($return_data['returnDelivery'])) {
            echo '<p><strong>Return Method:</strong> ' . esc_html($return_data['returnDelivery']['method'] ?? 'N/A') . '</p>';
            if (isset($return_data['returnDelivery']['trackingNumber'])) {
                echo '<p><strong>Tracking:</strong> ' . esc_html($return_data['returnDelivery']['trackingNumber']) . '</p>';
            }
        }
        echo '</div>';
        echo '</div>';

        // Show returned products with more detail
        echo '<div style="margin-bottom: 15px;">';
        echo '<h5>Returned Products:</h5>';
        echo '<table style="width: 100%; border-collapse: collapse;">';
        echo '<thead>';
        echo '<tr style="background: #f0f0f0;">';
        echo '<th style="padding: 8px; text-align: left; border: 1px solid #ddd;">Product</th>';
        echo '<th style="padding: 8px; text-align: center; border: 1px solid #ddd;">Qty</th>';
        echo '<th style="padding: 8px; text-align: center; border: 1px solid #ddd;">Type</th>';
        echo '<th style="padding: 8px; text-align: right; border: 1px solid #ddd;">Unit Price</th>';
        echo '<th style="padding: 8px; text-align: right; border: 1px solid #ddd;">Line Total</th>';
        echo '<th style="padding: 8px; text-align: left; border: 1px solid #ddd;">Reason</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ($return_data['returnedProducts'] as $product) {
            $wc_product = wc_get_product($product['id']);
            $product_name = $wc_product ? $wc_product->get_name() : 'Product ID: ' . $product['id'];

            echo '<tr>';
            echo '<td style="padding: 8px; border: 1px solid #ddd;">' . esc_html($product_name) . '</td>';
            echo '<td style="padding: 8px; text-align: center; border: 1px solid #ddd;">' . esc_html($product['returnedQuantity']) . '</td>';
            echo '<td style="padding: 8px; text-align: center; border: 1px solid #ddd;">' . esc_html($product['returnType'] ?? 'N/A') . '</td>';
            echo '<td style="padding: 8px; text-align: right; border: 1px solid #ddd;">' . wc_price($product['unitPrice'] / 100) . '</td>';
            echo '<td style="padding: 8px; text-align: right; border: 1px solid #ddd;">' . wc_price($product['linePrice'] / 100) . '</td>';
            echo '<td style="padding: 8px; border: 1px solid #ddd;">' . esc_html($product['reasonLabels']['pl'] ?? 'N/A') . '</td>';
            echo '</tr>';

            // Add notes row if exists
            if (!empty($product['notes'])) {
                echo '<tr>';
                echo '<td colspan="6" style="padding: 8px; border: 1px solid #ddd; background: #f9f9f9; font-style: italic;">';  // Changed from 5 to 6
                echo '<strong>Notes:</strong> ' . esc_html($product['notes']);
                echo '</td>';
                echo '</tr>';
            }
        }
        echo '</tbody>';
        echo '</table>';
        echo '</div>';

        // Action buttons for pending returns
        if ($return['status'] === 'RECEIVED') {
            echo '<div class="return-actions">';
            echo '<button type="button" class="button button-primary openapp-button accept accept-return" 
          data-return-key="' . esc_attr($meta_key) . '" 
          data-order-id="' . esc_attr($order_id) . '"
          data-action="accept">';
            echo '✓ Accept Refund via OpenApp';
            echo '</button>';
            echo '<button type="button" class="button openapp-button reject reject-return" 
          data-return-key="' . esc_attr($meta_key) . '" 
          data-order-id="' . esc_attr($order_id) . '"
          data-action="reject">';
            echo '✗ Reject Refund Request';
            echo '</button>';
            echo '</div>';
        }

        echo '</div>';
    }


    private function get_order_returns($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return array();
        }

        $returns = array();
        $meta_data = $order->get_meta_data();

        foreach ($meta_data as $meta) {
            $key = $meta->get_data()['key'];
            if (strpos($key, '_openapp_return_') === 0) {
                $returns[$key] = $meta->get_data()['value'];
            }
        }

        return $returns;
    }

    /**
     * Add JavaScript for admin return actions
     */
    private function add_returns_admin_script() {
        ?>
        <script type="text/javascript">
            jQuery(document).ready(function($) {
                $('.accept-return, .reject-return').on('click', function(e) {
                    e.preventDefault();

                    var $button = $(this);
                    var returnKey = $button.data('return-key');
                    var orderId = $button.data('order-id');
                    var action = $button.data('action');
                    var originalText = $button.text();

                    // Define labels based on action
                    var labels = {
                        'accept': {
                            actionText: 'Accept',
                            successMessage: 'Accept successful!',
                            processingLabel: 'Processing...'
                        },
                        'reject': {
                            actionText: 'Reject',
                            successMessage: 'Reject successful!',
                            processingLabel: 'Processing...'
                        }
                    };

                    var currentLabels = labels[action];
                    var notes = prompt('Enter notes for this ' + currentLabels.actionText.toLowerCase() + ' action:');
                    if (notes === null) return; // User cancelled

                    $button.prop('disabled', true).text(currentLabels.processingLabel);

                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'openapp_handle_return_action',
                            return_key: returnKey,
                            order_id: orderId,
                            return_action: action,
                            notes: notes,
                            nonce: '<?php echo wp_create_nonce('openapp_return_action'); ?>'
                        },
                        success: function(response) {
                            if (response.success) {
                                alert(currentLabels.successMessage);
                                location.reload();
                            } else {
                                alert('Error: ' + response.data);
                                $button.prop('disabled', false).text(originalText);
                            }
                        },
                        error: function() {
                            alert('An error occurred. Please try again.');
                            $button.prop('disabled', false).text(originalText);
                        }
                    });
                });
            });
        </script>
        <?php
    }

    public function handle_return_action_ajax() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'openapp_return_action')) {
            wp_send_json_error('Invalid nonce');
        }

        // Check permissions
        if (!current_user_can('edit_shop_orders')) {
            wp_send_json_error('Insufficient permissions');
        }

        $return_key = sanitize_text_field($_POST['return_key']);
        $order_id = intval($_POST['order_id']);
        $action = sanitize_text_field($_POST['return_action']); // 'accept' or 'reject'
        $notes = sanitize_textarea_field($_POST['notes']);

        $order = wc_get_order($order_id);
        if (!$order) {
            wp_send_json_error('Order not found');
        }

        // Get return data - HPOS compatible
        $return_meta = $order->get_meta($return_key, true);
        if (!$return_meta) {
            wp_send_json_error('Return not found');
        }

        $new_status = ($action === 'accept') ? 'ACCEPTED' : 'REJECTED';

        // Update return status locally - HPOS compatible
        $return_meta['status'] = $new_status;
        $return_meta['notes'] = $notes;
        $return_meta['updated_at'] = current_time('mysql');
        $order->update_meta_data($return_key, $return_meta);
        $order->save();

        // Send update to OpenApp
        $oa_response = $this->update_return_status_in_openapp(
            $return_meta['oa_return_id'],
            $new_status,
            $notes,
            (string)$order_id
        );

        if (!$oa_response['success']) {
            // Revert local change if OpenApp update failed - HPOS compatible
            $return_meta['status'] = 'RECEIVED';
            unset($return_meta['notes']);
            unset($return_meta['updated_at']);
            $order->update_meta_data($return_key, $return_meta);
            $order->save();

            wp_send_json_error('Failed to update OpenApp: ' . $oa_response['error']);
        }

        // Process the return in WooCommerce
        if ($new_status === 'ACCEPTED') {
            $this->process_return_acceptance($order, $return_meta, $notes);
        } else {
            $this->process_return_rejection($order, $return_meta, $notes);
        }

        wp_send_json_success('Return ' . strtolower($new_status) . ' successfully');
    }
    private function process_return_acceptance($order, $return_meta, $notes) {
        $return_data = $return_meta['return_data'];

        // Check if auto-refund is enabled
        $auto_refund = $this->get_option('auto_refund_on_accept', 'yes');

        if ($auto_refund === 'yes') {
            $refund_result = $this->create_wc_refund($order, $return_data);

            if ($refund_result['success']) {
                $refund = $refund_result['refund'];
                $order->add_order_note(sprintf(
                    "Return accepted and refund #%d created. Amount: %s. Admin notes: %s",
                    $refund->get_id(),
                    wc_price($refund->get_amount()),
                    $notes
                ));
            } else {
                $order->add_order_note("Return accepted but refund creation failed: " . $refund_result['error'] . ". Manual processing required. Admin notes: " . $notes);
            }
        } else {
            $order->add_order_note("Return accepted. Manual refund processing required. Admin notes: " . $notes);
        }
    }

    private function process_return_rejection($order, $return_meta, $notes) {
        $order->add_order_note("Return rejected by admin. Reason: " . $notes);
    }

    private function create_wc_refund($order, $return_data) {
        try {
            // Calculate refund amount and line items
            $refund_amount = 0;
            $line_items = array();

            foreach ($return_data['returnedProducts'] as $returned_product) {
                $product_id = $returned_product['id'];
                $returned_qty = $returned_product['returnedQuantity'];
                $line_price = $returned_product['linePrice'] / 100; // Convert from cents

                $refund_amount += $line_price;

                // Find the order item for this product
                foreach ($order->get_items() as $item_id => $item) {
                    if ($item->get_product_id() == $product_id || $item->get_variation_id() == $product_id) {
                        $line_items[$item_id] = array(
                            'qty' => $returned_qty,
                            'refund_total' => $line_price,
                            'refund_tax' => 0 // You might want to calculate tax here
                        );
                        break;
                    }
                }
            }

            // Create the refund
            $refund = wc_create_refund(array(
                'order_id' => $order->get_id(),
                'amount' => $refund_amount,
                'reason' => 'OpenApp Return: ' . $return_data['type'] . ' (Return ID: ' . $return_data['oaOrderReturnId'] . ')',
                'line_items' => $line_items
            ));

            if (is_wp_error($refund)) {
                return array(
                    'success' => false,
                    'error' => $refund->get_error_message()
                );
            }

            return array(
                'success' => true,
                'refund' => $refund
            );

        } catch (Exception $e) {
            return array(
                'success' => false,
                'error' => $e->getMessage()
            );
        }
    }



    private function update_return_status_in_openapp($oa_return_id, $status, $notes, $shop_order_id) {
        if (!$oa_return_id) {
            return array(
                'success' => false,
                'error' => 'Return ID is required'
            );
        }

        // Validate status according to API docs
        $valid_statuses = ['RECEIVED', 'ACCEPTED', 'REJECTED'];
        if (!in_array($status, $valid_statuses)) {
            return array(
                'success' => false,
                'error' => 'Status must be one of: ' . implode(', ', $valid_statuses)
            );
        }

        $endpoint = '/v1/returns/' . $oa_return_id;
        $method = 'PATCH';
        $url = $this->open_app_url . $endpoint;
        $context = 'openapp_update_return';

        $start_time = microtime(true);

        // Prepare data for OpenApp (all fields are required according to docs)
        $return_data = array(
            'status' => $status,
            'notes' => $notes,
            'shopOrderId' => $shop_order_id
        );

        $body = wp_json_encode($return_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        $setupData = $this->setupRequest($endpoint, $method, $body, $context);
        $response = $this->executeRequest($url, $setupData, $context);

        $end_time = microtime(true);
        $execution_time = round($end_time - $start_time, 4);
        $this->log_debug("update_return_status_in_openapp: " . $execution_time);

        // Process the actual response
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => $response->get_error_message()
            );
        }

        $http_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        // For PATCH /v1/returns, we expect HTTP 204 (No Content) on success
        if ($http_code === 204) {
            return array(
                'success' => true,
                'status_code' => 204,
                'body' => null,
                'raw_body' => $response_body
            );
        } else {
            // Handle error responses
            $error_message = "HTTP {$http_code}";
            if (!empty($response_body)) {
                $error_data = json_decode($response_body, true);
                if ($error_data && isset($error_data['error'])) {
                    $error_message .= ": " . $error_data['error'];
                } else {
                    $error_message .= ": " . $response_body;
                }
            }

            return array(
                'success' => false,
                'error' => $error_message,
                'status_code' => $http_code,
                'raw_body' => $response_body
            );
        }
    }

    /**
     * ===========================================================================
     * 11. Render QR codes
     * ===========================================================================
     */

    public function openapp_qr_order($atts = array()){
        $sanitized_uri = filter_var($_SERVER['REQUEST_URI'], FILTER_SANITIZE_URL);
        $this->log_debug('openapp_qr_order triggered at ' . gmdate('Y-m-d H:i:s') . ' by ' . $sanitized_uri);


        // parse the shortcode attributes
        $args = shortcode_atts(array(
            'lang' => sanitize_text_field(get_bloginfo('language')),
        ), $atts, 'openapp_qr_order');

        wp_enqueue_script('openappgw-js');

        global $woocommerce;
        $sessionId = $this->get_session_id();
        $cart_is_empty = $woocommerce->cart->is_empty();

        // Enqueue the script (only if it hasn't been enqueued already)
        if(!wp_script_is('openappgw-js-2', 'enqueued')) {
            wp_enqueue_script('openappgw-js-2', OPENAPPGW_PLUGIN_DIR_URL . 'assets/js/openapp-shortcodes-1.js', array('jquery'), OPENAPPGW_WOOCOMMERCE_GATEWAY, true);

            wp_enqueue_style('openapp-css', OPENAPPGW_PLUGIN_DIR_URL . 'assets/css/openapp-shortcode.css', array(), OPENAPPGW_WOOCOMMERCE_GATEWAY);

            // Pass PHP variables to JavaScript
            $localize_data = array(
                'baseUrl' => $this->get_base_url(),
                'cartId' => $this->get_cart_id_by_session($sessionId),
                'errorTextMessage' => __('OpenApp QR Code. Your cart is empty!','openapp-gateway-for-woocommerce'),
                'cartIsEmpty' => $cart_is_empty,
                'intervalTime' => $this->get_option('interval_time'),
                'sseEnabled' => false
            );

            if($this->is_sse_enabled()){
                $localize_data['sseEnabled'] = true;
            }

            wp_localize_script('openappgw-js-2', 'openappVars', $localize_data);
        }

        ob_start(); // Start capturing output into a buffer
        // Your CSS
        $output = '';

        $output .= '<div class="OpenAppCheckout-loading OpenAppCheckoutOrder" data-letter="C" data-style=""
data-merchantId="" data-integrationProfileId="" data-basketId="" data-basketValue="" data-basketCurrency="" data-uniqueProductsCount="" 
data-lang="'. $args['lang'] .'"
></div>';


       if(is_null($sessionId)){
           return "";
       }

        ob_end_clean(); // End capture and discard buffer

        $output = $this->minify_string($output);

        return $output; // Shortcodes in WordPress MUST return content
    }


    private function minify_string($string) {
        // Split the string into an array of lines
        $lines = explode("\n", $string);

        // Trim whitespace from each line
        $trimmed_lines = array_map('trim', $lines);

        // Filter out empty lines
        $non_empty_lines = array_filter($trimmed_lines);

        // Join the lines back into a single string
        return implode(' ', $non_empty_lines);
    }

    public function openapp_qr_login($atts = array()){
        $sanitized_uri = filter_var($_SERVER['REQUEST_URI'], FILTER_SANITIZE_URL);
        $this->log_debug('openapp_qr_login triggered at ' . gmdate('Y-m-d H:i:s') . ' by ' . $sanitized_uri);

        if (is_user_logged_in()) {
           return '<p><strong>OA login:</strong> user already login <a href="' . wp_logout_url( home_url()) .'" title="Logout">'. __('Logout'). '</a></p>';
        }

        if ( is_admin() ) {
            return;
        }

        // parse the shortcode attributes
        $args = shortcode_atts(array(
            'lang' => sanitize_text_field(get_bloginfo('language')),
        ), $atts, 'openapp_qr_login');

        wp_enqueue_script('openappgw-js');

        $sessionId = $this->get_session_id();
        $token = $this->get_cart_id_by_session($sessionId);
        $nonce = wp_create_nonce('wp_rest');

        // Enqueue the script (only if it hasn't been enqueued already)
        if(!wp_script_is('openappgw-js-3', 'enqueued')) {
            wp_enqueue_script('openappgw-js-3', OPENAPPGW_PLUGIN_DIR_URL . 'assets/js/openapp-shortcodes-2.js', array('jquery'), OPENAPPGW_WOOCOMMERCE_GATEWAY, true);

            wp_enqueue_style('openapp-css', OPENAPPGW_PLUGIN_DIR_URL . 'assets/css/openapp-shortcode.css', array(), OPENAPPGW_WOOCOMMERCE_GATEWAY);

            // Pass PHP variables to JavaScript
            $localize_data = array(
                'baseUrl' => esc_url($this->get_base_url()),
                'cartId' => sanitize_text_field($token),
                'errorTextMessage' => esc_html__('OpenApp QR Login. Something went wrong.', 'openapp-gateway-for-woocommerce')
            );
            wp_localize_script('openappgw-js-3', 'openappVars2', $localize_data);
        }

        $sessionActive = $this->user_session_is_active($token);

        if(!$sessionActive || is_null($token)){
            $activateLoginButtonHtml = '';

            $loginIcon = '<svg xmlns="http://www.w3.org/2000/svg" class="icon icon-tabler icon-tabler-login" width="24" height="24" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
  <path stroke="none" d="M0 0h24v24H0z" fill="none"/>
  <path d="M14 8v-2a2 2 0 0 0 -2 -2h-7a2 2 0 0 0 -2 2v12a2 2 0 0 0 2 2h7a2 2 0 0 0 2 -2v-2" />
  <path d="M20 12h-13l3 -3m0 6l-3 -3" />
</svg>';

            $activateLoginButtonHtml .= '<p><a href="/?oa-set-guest-session=1" class="OpenAppButton" id="jsOaCheckSession2">' . __('Login via OpenApp') . $loginIcon . '</a></p>';

            return $activateLoginButtonHtml;
        }

        ob_start(); // Start capturing output into a buffer

        $output = '';

        $output .= '<div class="OpenAppCheckout-loading OpenAppCheckoutLogin" data-letter="I" data-version="1" data-merchantId="" data-integrationProfileId="" data-token="" data-lang="'. $args['lang'] .'"></div>';

        ob_end_clean(); // End capture and discard buffer
        return $output; // Shortcodes in WordPress MUST return content
    }


    /**
     * ===========================================================================
     * 12. Functions for: oa_woocommerce_persistent_cart
     * ===========================================================================
     */

    private function get_product_object($item){
       if(is_null($item)){
           return null;
       }

        if ($item['variation_id'] > 0) {
            $_product = wc_get_product($item['variation_id']);
        } else {
            $_product = wc_get_product($item['product_id']);
        }

        return $_product;
    }

    private function get_simplified_cart_contents($my_cart) {
        $simplified_cart_contents = [];

        foreach ($my_cart as $cart_item_key => $cart_item) {
            // Copy the cart item, but exclude the 'data' field
            $simplified_cart_item = $cart_item;
            unset($simplified_cart_item['data']);
            $simplified_cart_contents[$cart_item_key] = $simplified_cart_item;
        }

        return $simplified_cart_contents;
    }

    function filterProductDataProperty($product) {
        $productName = $product->get_name();
        $shippingClass = $product->get_shipping_class();
        $weight = $product->get_weight();
        $needsShipping = $product->needs_shipping();
        $customProduct = array(
            'name' => $productName,
            'shippingClass' => $shippingClass,
            'weight' => $weight,
            'needsShipping' => $needsShipping
        );
        return $customProduct;
    }

    private function get_cart_contents_data($cart_contents) {
        // error_log(wp_json_encode($cart_contents, JSON_PRETTY_PRINT));

        $cart_contents_data = [];

        foreach ($cart_contents as $cart_item_key => $cart_item) {
            if (isset($cart_item['data'])) {
                $cart_contents_data[$cart_item_key] = $this->filterProductDataProperty($cart_item['data']);
            }
        }

        return $cart_contents_data;
    }

    public function get_cart_id_by_session($session_id) {
        global $wpdb;

        if(is_null($session_id)){
            return null;
        }

        // $this->log_debug("SESSION_ID:".$session_id);

        // Hashing session ID
        $hashed_session_id = hash('md5', $session_id);

        // Preparing SQL query
        $cart_id = $wpdb->get_var($wpdb->prepare("SELECT cart_id FROM {$wpdb->prefix}oa_woocommerce_persistent_cart WHERE cart_session_id = %s", $hashed_session_id));


        if(is_null($cart_id)){
            // cart is empty.. @TODO ?
            // return $hashed_session_id;
        }

        return $cart_id;
    }


    private function get_basket_data($cart_id) {
        $cart_contents_record = $this->get_cart_contents_record($cart_id);
        if (empty($cart_contents_record)) {
            return array();
        }

        $products = $this->get_products_from_cart_contents($cart_contents_record['cart_contents']);
        $total = $this->calculate_total_price($products);
        $discounts = $this->get_discounts($cart_contents_record['coupon_data']);
        $deliveryOptions = $this->get_available_shipping_methods($this->supported_country, $total, $cart_contents_record);

        $currency = get_woocommerce_currency();

        $basket_data = array(
            'expiresAt' => gmdate('Y-m-d\TH:i:s\Z', strtotime('+1 day')),
            'price' => array(
                'currency' => $currency,
                'basketValue' => round($total, 2),
                'discounts' => $discounts
            ),
            'deliveryOptions' => $deliveryOptions,
            'products' => $products,
            'loggedUser' => $cart_contents_record['user_id']
        );

        $response = array(
            'id' => $cart_id,
        );
        return array_merge($response, $basket_data);
    }

    private function get_cart_contents_record($cart_id) {
        global $wpdb;

        $cart_data = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM " . $wpdb->prefix . "oa_woocommerce_persistent_cart WHERE cart_id = %s", $cart_id)
        );

        return is_null($cart_data) ? array() : maybe_unserialize($cart_data->cart_contents);
    }

    private function calculate_total_price($products) {
        $total_in_cents = 0;

        foreach ($products as $item) {
            // Check if this is WooCommerce cart contents (has line_total and line_tax)
            if (isset($item['line_total']) && isset($item['line_tax'])) {
                // Convert each component to cents separately, then add
                // This maintains precision for each tax/total calculation
                $line_total_cents = $this->convertToCents($item['line_total']);
                $line_tax_cents = $this->convertToCents($item['line_tax']);
                $total_in_cents += ($line_total_cents + $line_tax_cents);
            }
            // Check for your custom linePrice field (already in cents)
            elseif (isset($item['linePrice'])) {
                // Ensure it's an integer to avoid float precision issues
                $total_in_cents += (int) $item['linePrice'];
            }
            // Fallback to line_total only
            elseif (isset($item['line_total'])) {
                $total_in_cents += $this->convertToCents($item['line_total']);
            }
        }

        // Return total in cents (ensure integer)
        return (int) $total_in_cents;
    }


    private function get_products_from_cart_contents($cart_contents) {
        $products = array();
        foreach ($cart_contents as $item) {
            $product_data = $this->create_product_output_array_from_cart($item);
            if (!empty($product_data)) {
                $products[] = $product_data;
            }
        }
        return $products;
    }


    /**
     * @param WP_REST_Request $request
     * @return WP_REST_Response|null
     */
    public function get_qr_code_data(WP_REST_Request $request){
        $cart_id = $request->get_param('cart_id');

        if(empty($cart_id)){
            $cart_data = array(
                'merchant_id' => $this->merchant_id,
                'profile_id' => $this->profile_id,
                'cart_id' => $cart_id,
                'total_value' => 0,
                'currency' => '',
                'unique_products_count' => '',
            );
            $wpRestResponse = new WP_REST_Response( $cart_data, 200 );
            return $wpRestResponse;
        }

        global $wpdb;

        // Retrieve cart contents
        $serialized_cart_contents = $wpdb->get_var($wpdb->prepare("SELECT cart_contents FROM {$wpdb->prefix}oa_woocommerce_persistent_cart WHERE cart_id = %s", $cart_id));


        if (!$serialized_cart_contents) {
            // for login - we allow empty cart
            // return null;
        }

        // Unserialize the cart contents
        $cart_data = maybe_unserialize($serialized_cart_contents);

        if(empty($cart_data)){
            return null;
        }

        $cart_contents = $cart_data['cart_contents'];

        if(empty($cart_contents)){
            // $this->store_cart_in_db(true);
            // for login - we allow empty cart
            // return null;
        }

        // Initialize values
        $unique_products_count = 0;

        $products = array();
        foreach($cart_contents as $item_key => $item) {
            $product_data = $this->create_product_output_array_from_cart($item);
            if(!empty($product_data)){
                array_push($products, $product_data);
                $unique_products_count++;
            }
        }

        $total_value = $this->calculate_total_price($products);


        // Get currency - assuming all products in the cart have the same currency
        $currency = get_woocommerce_currency();

        $cart_data = array(
            'merchant_id' => $this->merchant_id,
            'profile_id' => $this->profile_id,
            'cart_id' => $cart_id,
            'total_value' => $total_value,
            'currency' => $currency,
            'unique_products_count' => $unique_products_count,
        );

        // wp_send_json($cart_data);
        $wpRestResponse = new WP_REST_Response( $cart_data, 200 );
        return $wpRestResponse;

    }


    /**
     * ===========================================================================
     * 13. HELPERS
     * ===========================================================================
     */

    private function get_base_url() {
        return rtrim(site_url(), '/');
    }

    private function openappgw_custom_log($message, $context = ''){
        if ( function_exists( 'openappgw_custom_log' ) ) {
            openappgw_custom_log($message, $context);
        }
    }

    private function get_values_and_types($data) {
        if(is_array($data) || is_object($data)) {
            foreach($data as $key => $value) {
                if(is_array($value) || is_object($value)) {
                    $data[$key] = $this->get_values_and_types($value); // Recursive call for nested array or objects
                } else {
                    $data[$key] = array('value' => $value, 'type' => gettype($value));
                }
            }
        } else {
            $data = array('value' => $data, 'type' => gettype($data));
        }
        return $data;
    }

    private function destroy_user_sessions( $user_id ) {
        $sessions = WP_Session_Tokens::get_instance( $user_id );
        $sessions->destroy_all();
    }

    private function get_session_id(){
        $sessionId = null;
        if (isset(WC()->session)) {
            $sessionId = WC()->session->get_customer_id();
        }

        return $sessionId;
    }

    private function createResponse($data, $status = 200)
    {
        return new WP_REST_Response($data, $status);
    }

    /** @TODO */
    private function validate_discount($discount) {
        // Implement your validation logic here...
        // You could check if a coupon with the given code exists,
        // if it's still valid (not expired), etc.

        // For now, let's say all discounts are valid
        return true;
    }


    protected function encodeJsonResponse($response)
    {
        // return json_encode($response);
        return wp_json_encode($response, JSON_UNESCAPED_SLASHES);
    }


    private function create_product_output_array($_product, $quantity, $error = null) {
        $product_url = wp_get_attachment_url($_product->get_image_id());

        $unit_price = round($_product->get_price() * 100);
        $line_price = round($_product->get_price() * $quantity * 100);
        $originalUnitPrice = round($_product->get_price() * 100);
        $originalLinePrice = round($_product->get_price() * $quantity * 100);

        $product_data =  array(
            'ean' => $_product->get_sku(),
            'id' => (string)$_product->get_id(),
            'name' => $_product->get_name(),
            'images' => array($product_url),
            'quantity' => $quantity,
            'unitPrice' => $unit_price,
            'linePrice' => $line_price,
            'originalUnitPrice' => $originalUnitPrice,
            'originalLinePrice' => $originalLinePrice
        );
        if($error){
            $product_data['error'] = $error;
        }

        return $product_data;

    }

    private function create_product_output_array_from_cart($item) {
        // Important! Return main product or variation
        $_product = $this->get_product_object($item);

        if(is_null($_product)){
            return array();
        }

        $product_url = wp_get_attachment_url($_product->get_image_id());

        // Unit prices before discount
        $original_unit_price_exclusive = $item['line_subtotal'] / $item['quantity'];
        $original_unit_tax = $item['line_subtotal_tax'] / $item['quantity'];
        $original_unit_price_inclusive = $original_unit_price_exclusive + $original_unit_tax;

        // Unit prices after discount
        $unit_price_exclusive = $item['line_total'] / $item['quantity'];
        $unit_tax = $item['line_tax'] / $item['quantity'];
        $unit_price_inclusive = $unit_price_exclusive + $unit_tax;

        // Line prices (for all quantities)
        $line_price_exclusive = $item['line_total'];
        $line_tax = $item['line_tax'];
        $line_price_inclusive = $line_price_exclusive + $line_tax;

        // Original line prices (for all quantities) before discount
        $original_line_price_exclusive = $item['line_subtotal'];
        $original_line_tax = $item['line_subtotal_tax'];
        $original_line_price_inclusive = $original_line_price_exclusive + $original_line_tax;

        $product_data = array(
            'ean' => $_product->get_sku(),
            'id' => (string)$_product->get_id(),
            'name' => $_product->get_name(),
            'images' => array($product_url),
            'quantity' => $item['quantity'],
            'unitPrice' => round($unit_price_inclusive * 100),  // Converted to grosz
            'linePrice' => round($line_price_inclusive * 100),
            'originalUnitPrice' => round($original_unit_price_inclusive * 100),
            'originalLinePrice' => round($original_line_price_inclusive * 100),
        );

        return $product_data;
    }

    private function get_delivery_options() {
        $methods = $this->shipping_methods;

        $enabled_methods = array();

        foreach ($methods as $key => $method) {
            if ('yes' === get_option('woocommerce_openapp_' . $key . '_enabled')) {
                $enabled_methods[] = array(
                    'key' => $key,
                    // convert cost from decimal format to minor currency unit (cents)
                    'cost' => (int) (get_option('woocommerce_openapp_' . $key . '_cost') * 100)
                );
            }
        }

        return $enabled_methods;
    }


    private function get_discounts($coupons) {
        $discounts = array();

        foreach($coupons as $coupon) {
            $discounts[] = array(
                'code' => $coupon['code'],
                'value' => round($coupon['discount_amount'] * 100)
            );
        }

        return $discounts;
    }

    private function user_session_is_active($token){

        $woocommerceSessionActive = false;

        if (isset(WC()->session)) {
            if (WC()->session->has_session()) {
                $woocommerceSessionActive = true;
            }
        }

        if($woocommerceSessionActive && !is_null($token)){
            return true;
        }

        return false;
    }

    private function is_heartbeat() {
        $action = filter_input(INPUT_POST, 'action', FILTER_SANITIZE_SPECIAL_CHARS);
        return ($action == 'heartbeat');
    }

    private function base64Encode($data) {
        return base64_encode($data);
    }

    private function createWpUser($email, $random_password) {
        $user_id = wc_create_new_customer($email, '', $random_password);

        if (is_wp_error($user_id)) {
            // Handle the error.
            //echo 'Unable to create a new user: ' . $user_id->get_error_message();
            // return $user_id;
        }

        return $user_id;
    }

    /**
     * ===========================================================================
     * 14. TESTING
     * ===========================================================================
     */

    public function log_debug($message) {
        if ('yes' === $this->get_option('debug', 'no')) {
            if (is_array($message)) {
                $message = print_r($message, true);
            }
            $timestamp = gmdate('Y-m-d H:i:s');
            error_log("[OpenApp {$timestamp}] " . $message);
        }
    }

    private function mask_secret($secret){
        $start = substr($secret, 0, 4); // Get the first four characters
        $end = substr($secret, -4); // Get the last four characters
        $masked = str_repeat('*', strlen($secret) - 8); // Generate a string of '*' with the same length as the remaining characters

        return $start . $masked . $end; // Return the masked secret
    }


}
