<?php
require_once dirname(__FILE__) . '/../OpenappAbstractRESTController.php';

use Symfony\Component\HttpFoundation\Request;

/**
 * This REST endpoint displays order information
 */
class ps_openapporderModuleFrontController extends OpenappAbstractRESTController
{
    public $id_customer;
    public $id_address;
    public $user_id;

    private $placeholder = 'xxxxxxxx';
    private $placeholder2 = 'N/A';

    protected function processPostRequest()
    {
        try {
        $bodyRaw = Tools::file_get_contents('php://input');
        $_POST           = json_decode($bodyRaw, true);
        $oaOrderId       = Tools::getValue('oaOrderId');
        $basket          = Tools::getValue('basket');
        $billingDetails  = Tools::getValue('billingDetails');
        $deliveryDetails = Tools::getValue('deliveryDetails');
        $paymentDetails = Tools::getValue('paymentDetails');

        $request = Request::createFromGlobals();
        $headers = $request->headers->all();


        /**
         * Log body request
         */

        $this->ct_custom_log("placeOrder_body:");
        $json_post_data = json_encode($_POST, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $this->ct_custom_log($json_post_data);

        /**
         * Set default response headers
         */
        header('x-server-authorization: aaabbbccc');
        header('Content-Type: application/json; charset=utf-8');

        // Disable caching
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Pragma: no-cache');
        header('Expires: Thu, 01 Jan 1970 00:00:00 GMT'); // Set an expired date in the past

        if (empty($oaOrderId)) {
            $this->sendError('missing_oaOrderId', $this->translator->trans('OA Order ID is required', [], 'Modules.Openapp.Errors'), 400);
        }


        $bodyHash = hash('sha256', $bodyRaw, true);
        $responseBodyHashBase64 = base64_encode($bodyHash);

        $validHmac = $this->isRequestValid($headers, $responseBodyHashBase64);
        if(!$validHmac){
            $this->sendError('invalid_auth', $this->translator->trans('Unauthorized request', [], 'Modules.Openapp.Errors'), 403);
        }


        // Verify if the JSON structure is as expected
        if (!is_array($_POST) || !isset($_POST['basket'], $_POST['deliveryDetails'], $_POST['paymentDetails'])) {
            $this->sendError('invalid_data', $this->translator->trans('Invalid JSON structure', [], 'Modules.Openapp.Errors'), 400);
        }

        // Validate basket structure after auth to mirror WP flow
        if (!is_array($basket) || !isset($basket['products']) || !isset($basket['id'])) {
            $this->sendError('invalid_data', $this->translator->trans('Invalid basket structure', [], 'Modules.Openapp.Errors'), 400);
        }

        $products        = $basket['products'];
        $cart_id         = $basket['id'];
        $lang_id         = (int) Configuration::get('PS_LANG_DEFAULT');
        $order_notes     = '';

        /**
         * OA request can have firstname and lastname empty
         */

        if(!isset($deliveryDetails['firstName'])){
            $deliveryDetails['firstName'] = $this->placeholder;
        }
        if(!isset($deliveryDetails['lastName'])){
            $deliveryDetails['lastName'] = $this->placeholder;
        }

        $loggedUser = isset($_POST['basket']['loggedUser']) ? $_POST['basket']['loggedUser'] : '0_0';
        $parts = explode('_', $loggedUser);
        $customerId = isset($parts[0]) ? $parts[0] : 0;
        $guestId = isset($parts[1]) ? $parts[1] : 0;


        $deliveryMethodKey = isset($deliveryDetails['method']) ? $deliveryDetails['method'] : null;

        $matchedCarrierId = null;
        $carriers = Carrier::getCarriers(Context::getContext()->language->id, true, false, false, null, Carrier::ALL_CARRIERS);
        foreach ($carriers as $carrier) {
            $key = 'CARRIER_MAP_OA_' . $carrier['id_carrier'];
            $carrierValue = Configuration::get($key);
            if ($carrierValue == $deliveryMethodKey) {
                $matchedCarrierId = $carrier['id_carrier'];
                break;
            }
        }

        $order_notes .= 'CustomerId: ' . $customerId . ' || GuestId: ' . $guestId . "\n";
        $order_notes .= "OA Order ID: " . $_POST['oaOrderId'] . "\n";
        $order_notes .= "Payment Currency: " . $_POST['paymentDetails']['currency'] . "\n";
        $order_notes .= "Payment Amount: " . $_POST['paymentDetails']['amount'] . "\n";
        $order_notes .= "Delivery Type: " . (isset($_POST['deliveryDetails']['type']) ? $_POST['deliveryDetails']['type'] : 'N/A') . "\n";
        $order_notes .= "Delivery Key: " . $deliveryMethodKey. "\n";
        $order_notes .= "Carrier_id: " . $matchedCarrierId. "\n";

        // Assuming you have the customer's email in the JSON
        $customer_email = $deliveryDetails['email'];

        // Create a new Customer instance
        $customer = new Customer();

        // Search for an existing customer by email
        $customer = $customer->getByEmail($customer_email);

        $is_new_customer = false;

        // Check if a customer with the provided email exists
        if ($customer && $customer->id) {
            // Customer exists, you can proceed with this customer object
        } else {
            // Create a new customer if it does not exist
            $customer = new Customer();
            $customer->email = $customer_email;
            $customer->firstname = $deliveryDetails['firstName'];
            $customer->lastname = $deliveryDetails['lastName'];
            $randomPassword = Tools::passwdGen(15);
            $customer->passwd = Tools::encrypt($randomPassword);
            $customer->secure_key = md5(uniqid(rand(), true));
            $is_new_customer = true;
        }

        // Validate customer data
        $retValidateFields = $customer->validateFields(false, true);
        if ($retValidateFields !== true) {
            $this->sendError('invalid_data', $retValidateFields, 400);
        }


        // Add or update the customer as needed
        if ($is_new_customer) {
            $customer->add();
        } else {
            // For an existing customer, ensure they have a secure_key
            if (!$customer->secure_key) {
                $customer->secure_key = md5(uniqid(rand(), true));
            }
            $customer->update();
        }

        $customerId = (int) $customer->id;


        $is_new_cart = false;

        // Create or find cart
        $cart = new Cart($cart_id);
        if (!$cart->id) {
            $cart = new Cart();
            $is_new_cart = true;
        }

        // cart can be created by guest user
        // having email in request - we can assign it to proper customer
        $cart->id_customer = $customerId;

        if($is_new_cart){
            $cart->id_currency = Context::getContext()->currency->id; // Set the currency from the current context
            $cart->id_lang = Context::getContext()->language->id; // Set the language from the current context
            $cart->id_shop = Context::getContext()->shop->id; // Set the shop ID from the current context
            $cart->add();
        } else {
            $cart->update();
        }



        /**
         * First - clear current presta cart products
         */
        $products_in_session = $cart->getProducts();
        foreach ($products_in_session as $product) {
           $cart->deleteProduct($product['id_product'], $product['id_product_attribute']);
        }
        $cart->update();

        /**
         * Handle delivery address
         */
        $deliveryAddress = $this->createAndValidateDeliveryAddress($customerId, $deliveryDetails, 'Delivery Address');


        // Check if the address already exists
        $existingDeliveryAddressId = $this->addressExistsForCustomer($customerId, $deliveryDetails, 'Delivery Address');

        if (!$existingDeliveryAddressId) {
            $deliveryAddress->add();
            $prestaDeliveryAddressId = $deliveryAddress->id;
        } else {
            $prestaDeliveryAddressId = $existingDeliveryAddressId;
        }

        $cart->id_address_delivery = $prestaDeliveryAddressId;

        /**
         * Handle billing address
         */
        if (!empty($billingDetails)) {

            /**
             * OA invoice can have firstname and lastname empty
             */
            if(!isset($billingDetails['firstName'])){
                $billingDetails['firstName'] = $this->placeholder2;
            }
            if(!isset($billingDetails['lastName'])){
                $billingDetails['lastName'] = $this->placeholder2;
            }

            $billingAddress = $this->createAndValidateBillingAddress($customerId, $billingDetails, 'Billing Address');

            // Check if the billing address already exists and add/update accordingly
            $existingBillingAddressId = $this->addressExistsForCustomer($customerId, $billingDetails, 'Billing Address');
            if (!$existingBillingAddressId) {
                $billingAddress->add();
                $billingAddressId = $billingAddress->id;
            } else {
                $billingAddressId = $existingBillingAddressId;
            }

            // Set the billing address ID to the cart
            $cart->id_address_invoice = $billingAddressId;
        } else {
            // If no separate billing details, use delivery address for billing
            $cart->id_address_invoice = $prestaDeliveryAddressId;
        }

        /**
         * update cart secure_key
         */
        $cart->secure_key = $customer->secure_key;
        // $cart->save();

        /**
         * assign carrier to the order
         */

        if ($matchedCarrierId !== null) {
            $carrier = new Carrier($matchedCarrierId);
            // Check if the carrier object is loaded correctly and is active
            if (Validate::isLoadedObject($carrier) && $carrier->active) {
                $cart->id_carrier = (int)$matchedCarrierId;
                $cart->update();
            }
        }

        /**
         * Add products
         */
        $productErrors = [];
        foreach ($products as $product) {
            // Split the concatenated ID to get productId and productAttributeId
            list($productId, $productAttributeId) = $this->parseProductId($product['id']);

            $quantity = $product['quantity'];
            $productToAdd = new Product($productId, false, $lang_id);

            if ($productId <= 0 || $quantity <= 0) {
                $productErrors[] = "Invalid product ID or quantity for item {$product['id']}.";
                continue;
            }

            // Check if product is loaded, active, and available for order
            if (!Validate::isLoadedObject($productToAdd) || !$productToAdd->active || !$productToAdd->available_for_order) {
                $productErrors[] = "Product ID {$productId} is not valid or available.";
                continue;
            }

            // Check if the required quantity is available
            $availableQuantity = ($productAttributeId > 0) ? Product::getQuantity($productId, $productAttributeId) : $productToAdd->checkQty($quantity);
            if ($availableQuantity < $quantity) {
                $productErrors[] = "Insufficient quantity for Product ID {$productId}.";
                continue;
            }

            // Add the product to the cart
            if ($productAttributeId > 0) {
                $cart->updateQty($quantity, $productId, $productAttributeId);
            } else {
                $cart->updateQty($quantity, $productId);
            }
        }

        if (!empty($productErrors)) {
            $this->sendError('invalid_data', implode(', ', $productErrors), 400);
        }



        /**
         * Order of this code is important (!)
         */
        $deliveryOption = [(int) $cart->id_address_delivery => $matchedCarrierId . ','];
        $cart->setDeliveryOption($deliveryOption);
        $cart->update();


        /**
         * @TODO
         * We are updating real prestashop cart database record
         * Keep in mind that ps_oa_persistent_cart - is not currently updated
         * Edge case: prestashop cart vs ps_oa_persistent_cart - will be unsynchronized (?))
         */


        if (!Validate::isLoadedObject($cart)) {
            $this->sendError('invalid_data', $this->translator->trans('Invalid cart structure', [], 'Modules.Openapp.Errors'), 400);
        }

        $order_id = null;
        $order_reference = null;

        if ($cart->id && $cart->OrderExists() == false) {
            // Use an existing payment module
            $module_name = 'ps_openapp';
            $module = Module::getInstanceByName($module_name);

            if ($module && $module->active) {
                $orderState = Configuration::get('PS_OS_PAYMENT'); // payment accepted
                if (!$orderState) {
                    $this->sendError('invalid_data', $this->translator->trans('Invalid order status', [], 'Modules.Openapp.Errors'), 400);
                }

                $amount_paid = floatval($paymentDetails['amount'] / 100);
                $order_notes = strip_tags($order_notes, '<br>');

                try {
                    $module->validateOrder(
                        $cart->id, // $id_cart
                        $orderState, // $id_order_state
                        $amount_paid, // $amount_paid, assuming amount is in cents
                        $module->displayName, // $payment_method, replace with the actual payment method name
                        null, // $message
                        [], // $extra_vars (additional variables, if any)
                        null, // $currency_special
                        false, // $dont_touch_amount
                        $customer->secure_key // $secure_key
                    );
                } catch (\Exception $e) {
                    $this->ct_custom_log("validateOrder Exception: " . $e->getMessage());
                    $this->sendError('order_validation_failed', $e->getMessage(), 500);
                }

                // Get the order ID from the module instance
                $order_id = $module->currentOrder;

                // Now, you can retrieve the order reference if the order ID is valid
                if ($order_id) {
                    $order = new Order($order_id);
                    $order_reference = $order->reference;

                    /**
                     * set payment transaction_id
                     */
                    $order_payments = $order->getOrderPaymentCollection();
                    if (!empty($order_payments)) {
                        foreach ($order_payments as $order_payment) {
                            if ($order_payment->amount == $amount_paid) {
                                $order_payment->transaction_id = $oaOrderId;
                                $order_payment->update();
                            }
                        }
                    }

                    /**
                     * Add payment info as private admin message
                     */
                    $paymentMessage = "=== OPENAPP PAYMENT COMPLETED ===\n" . $order_notes;
                    $this->addPrivateOrderMessage($order, $paymentMessage);

                    /**
                     * Update order_key in custom table - using safe SQL with proper escaping
                     */
                    $table_name = _DB_PREFIX_.'ps_oa_persistent_cart';
                    $sql = new DbQuery();
                    $sql->select('*');
                    $sql->from('ps_oa_persistent_cart');
                    $sql->where('cart_id = ' . (int)$cart_id);
                    $exec = Db::getInstance()->executeS($sql);

                    if($exec && count($exec) > 0) {

                        $order_count = (int) $exec[0]['order_count'] + 1;

                        $update_1 = 'UPDATE `' . $table_name . '` SET `order_count` = ' . (int)$order_count . ', `order_key` = \'' . pSQL($order_reference) . '\', `oaOrderId` = \'' . pSQL($oaOrderId) . '\' WHERE `cart_id` = ' . (int)$cart_id;
                        Db::getInstance()->execute($update_1);
                    }

                }
            } else {
                // Handle the case where the payment module is not found or not active
            }
        } else {
            // Handle the case where the order already exists
            $this->sendError('order_already_exists', $this->translator->trans('Order already exists', [], 'Modules.Openapp.Errors'), 400);
        }


        if(!is_null($order_id)) {
            /**
             * Order success
             */
            $max_return_days = 30;

            $response = array(
                "shopOrderId" => (string) $order_reference,
                "returnPolicy" => array(
                    "maxReturnDays" => $max_return_days,
                )
            );

            $expectedXServerAuth = $this->calculate_server_authorization($headers, $response);

            if($expectedXServerAuth !== null) {
                header('X-Server-Authorization: '.$expectedXServerAuth);
            }

            $this->ajaxRender($this->encodeJsonResponse($response));
            die;

        } else {
            $this->sendError('error_order_creation', $this->translator->trans('Error in Order Creation', [], 'Modules.Openapp.Errors'), 400);
        }
        } catch (\Exception $e) {
            $this->sendError('order_processing_error', 'Order processing failed: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Helper functions
     */

    private function createAndValidateDeliveryAddress($customerId, $deliveryDetails, $alias) {

        // Handle delivery and invoice addresses
        $deliveryAddress = new Address();
        $deliveryAddress->id_customer = $customerId;

        // Set other address properties from $deliveryDetails
        $deliveryAddress->alias = $alias;
        $deliveryAddress->firstname = $deliveryDetails['firstName'];
        $deliveryAddress->lastname = $deliveryDetails['lastName'];
        if(isset($deliveryDetails['companyName'])){
            $deliveryAddress->company = $deliveryDetails['companyName'];
        }
        $deliveryAddress->address1 = $deliveryDetails['street'] . " " . $deliveryDetails['streetNo'];

        if(!empty($deliveryDetails['apartmentNo'])){
            $deliveryAddress->address1 .= ' / ' . $deliveryDetails['apartmentNo'];
        }

        if(isset($deliveryDetails['id'])){
            $deliveryAddress->address2 = $deliveryDetails['id'];
        }
        $deliveryAddress->postcode = $deliveryDetails['postalCode']; // Postal/Zip code
        $deliveryAddress->city = $deliveryDetails['city']; // City
        $deliveryAddress->phone = $deliveryDetails['phoneNumber']; // Phone number
        $deliveryAddress->phone_mobile = $deliveryDetails['phoneNumber']; // Mobile phone number, if different

        // Validate country code before assignment
        $countryId = Country::getByIso($deliveryDetails['country']);
        if (!$countryId) {
            $this->sendError('invalid_data', 'Invalid delivery country code: ' . $deliveryDetails['country'], 400);
        }
        $deliveryAddress->id_country = $countryId;

        $retValidateFields = $deliveryAddress->validateFields(false, true);

        if($retValidateFields !== true){
            $this->sendError('invalid_data', $retValidateFields, 400);
        }

        return $deliveryAddress;
    }

    private function createAndValidateBillingAddress($customerId, $billingDetails, $alias) {
        $billingAddress = new Address();
        $billingAddress->id_customer = $customerId;
        // Set other billing address properties from $billingDetails
        $billingAddress->alias = $alias;
        if(!empty($billingDetails['companyName'])) {
            $billingAddress->company = $billingDetails['companyName'];
        }
        if(!empty($billingDetails['firstName'])){
            $billingAddress->firstname = $billingDetails['firstName'];
        }
        if(!empty($billingDetails['lastName'])){
            $billingAddress->lastname = $billingDetails['lastName'];
        }

        $billingAddress->address1 = $billingDetails['street'] . " " . $billingDetails['streetNo'];

        if(!empty($billingDetails['apartmentNo'])){
            $billingAddress->address1 .= ' / ' . $billingDetails['apartmentNo'];
        }

        $billingAddress->postcode = $billingDetails['postalCode'];
        $billingAddress->city = $billingDetails['city'];
        $billingAddress->vat_number = $billingDetails['taxId'];

        // Validate country code before assignment
        $billingCountryId = Country::getByIso($billingDetails['country']);
        if (!$billingCountryId) {
            $this->sendError('invalid_data', 'Invalid billing country code: ' . $billingDetails['country'], 400);
        }
        $billingAddress->id_country = $billingCountryId;
        $billingAddress->other = isset($billingDetails['notes']) ? $billingDetails['notes'] : '';

        // Validate and add/update billing address
        $retValidateFields = $billingAddress->validateFields(false, true);
        if($retValidateFields !== true){
            $this->sendError('invalid_data', $retValidateFields, 400);
        }

        return $billingAddress;
    }

    private function addressExistsForCustomer($customer_id, $requestAddress, $alias) {
        $sql = new DbQuery();
        $sql->select('id_address');
        $sql->from('address', 'a');
        $sql->where('a.id_customer = ' . (int)$customer_id);
        $sql->where('a.alias = \'' . pSQL($alias) . '\'');
        if(!empty($requestAddress['firstName'])){
            $sql->where('a.firstname = \'' . pSQL($requestAddress['firstName']) . '\'');
        }
        if(!empty($requestAddress['lastName'])){
            $sql->where('a.lastname = \'' . pSQL($requestAddress['lastName']) . '\'');
        }
        if(!empty($requestAddress['companyName'])){
            $sql->where('a.company = \'' . pSQL($requestAddress['companyName']) . '\'');
        }

        // Match address1 with concatenated street and streetNo
        $address1 = pSQL($requestAddress['street']) . " " . pSQL($requestAddress['streetNo']);
        if (!empty($requestAddress['apartmentNo'])) {
            $address1 .= ' / ' . pSQL($requestAddress['apartmentNo']);
        }

        $sql->where('a.address1 = \'' . $address1 . '\'');

        // Match address2 with 'id' if it exists
        if (isset($requestAddress['id']) && !empty($requestAddress['id'])) {
            $address2 = pSQL($requestAddress['id']);
            $sql->where('a.address2 = \'' . $address2 . '\'');
        }

        $sql->where('a.postcode = \'' . pSQL($requestAddress['postalCode']) . '\'');
        $sql->where('a.city = \'' . pSQL($requestAddress['city']) . '\'');
        if(!empty($requestAddress['phoneNumber'])){
            $sql->where('a.phone = \'' . pSQL($requestAddress['phoneNumber']) . '\'');
        }
        $sql->where('a.id_country = ' . (int)Country::getByIso($requestAddress['country']));

        $id_address = Db::getInstance()->getValue($sql);
        return $id_address ? (int)$id_address : false;
    }


    private function displayCartDetails($cart) {
        $products = $cart->getProducts();

        echo "Cart ID: " . $cart->id . "\n";
        echo "Customer ID: " . $cart->id_customer . "\n";
        echo "Products in Cart:\n";
        echo "---\n";

        foreach ($products as $product) {
            $this->displayProductDetails($product);
        }

        echo "Total Products: " . $cart->nbProducts() . "\n";
        echo "Cart Total: " . $cart->getOrderTotal(true, Cart::BOTH) . "\n";
    }

    private function displayProductDetails($product) {
        $productId = $product['id_product'];
        $productAttributeId = $product['id_product_attribute'];
        echo "Product ID: " . $product['id_product'] . "\n";
        $price = Product::getPriceStatic($productId, true, $productAttributeId);
        $name = $product['name'];

        echo "Product attr ID: " . $productAttributeId . "\n";

        if ($productAttributeId > 0) {
            $combination = new Combination($productAttributeId);
            if (Validate::isLoadedObject($combination)) {
                $attributes = $combination->getAttributesName(Context::getContext()->language->id);
                $attributesString = implode(', ', array_column($attributes, 'name'));
                echo "Variant Details: " . $attributesString . "\n";
                $name = $name . ' - ' . $attributesString;

                // Get the specific price for the combination
                $price = Product::getPriceStatic($product['id_product'], true, $productAttributeId);
            }
        }

        echo "Product Name: " . $name . "\n";
        echo "Quantity: " . $product['cart_quantity'] . "\n";
        echo "Price: " . $price . "\n"; // Display the correct price
        echo "---\n";
    }

}
